<?php

declare(strict_types=1);

namespace IssetBV\MailBundle\Service\Mail\Center\Doctrine;

use Doctrine\Common\Persistence\ObjectManager;
use IssetBV\LogBundle\Log\LoggerTrait;
use IssetBV\MailBundle\Repository\MailRepository;
use IssetBV\MailBundle\Service\Mail\Center\Exception\MailNotFoundException;
use IssetBV\MailBundle\Service\Mail\Center\MailCenterInterface;
use IssetBV\MailBundle\Service\Mail\Center\Mailer\MailerInterface;
use IssetBV\MailBundle\Service\Mail\MailInterface;
use Psr\Log\LoggerAwareInterface;

class MailCenterDoctrine implements MailCenterInterface, LoggerAwareInterface
{
    use LoggerTrait;

    /**
     * @var MailerInterface
     */
    private $mailer;
    /**
     * @var MailToEntityConverter
     */
    private $mailToEntityConverter;
    /**
     * @var ObjectManager
     */
    private $manager;
    /**
     * @var MailRepository
     */
    private $mailRepository;
    /**
     * @var MailAddressStateChanger
     */
    private $mailAddressStateChanger;

    public function __construct(
        ObjectManager $manager,
        MailerInterface $mailer,
        MailToEntityConverter $mailToEntityConverter,
        MailRepository $mailRepository,
        MailAddressStateChanger $mailAddressStateChanger
    ) {
        $this->mailer = $mailer;
        $this->mailToEntityConverter = $mailToEntityConverter;
        $this->manager = $manager;
        $this->mailRepository = $mailRepository;
        $this->mailAddressStateChanger = $mailAddressStateChanger;
    }

    public function add(MailInterface $mail): string
    {
        $mailEntity = $this->mailToEntityConverter->convert($mail);
        $this->manager->persist($mailEntity);
        $this->manager->flush();
        try {
            $this->mailer->send(new MailChangeable($this->mailAddressStateChanger, $mailEntity));
            $this->manager->flush();
        } catch (\Exception $e) {
            $this->error('failed to send mail with uuid: [' . $mailEntity->getUuid() . '] exception: [' . $e->getMessage() . ']');
        }

        return $mailEntity->getUuid();
    }

    /**
     * @param string $uuid
     *
     * @throws MailNotFoundException
     *
     * @return MailInterface
     */
    public function get(string $uuid): MailInterface
    {
        $mail = $this->mailRepository->fetchByUUID($uuid);
        if (null === $mail) {
            throw new MailNotFoundException('Mail not found: ' . $uuid);
        }

        return new MailProxy($mail);
    }
}
