<?php

declare(strict_types=1);

namespace IssetBV\MailBundle\Service\Mail\Center\Mailer\Swift;

use Doctrine\Common\Persistence\ObjectManager;
use IssetBV\LogBundle\Log\LoggerTrait;
use IssetBV\MailBundle\Service\Mail\Center\Doctrine\MailAddressStateChanger;
use Psr\Log\LoggerAwareInterface;
use Swift_Events_SendEvent;
use Swift_Events_SendListener;
use Swift_Events_TransportExceptionEvent;
use Swift_Events_TransportExceptionListener;
use Swift_Mime_Message;

class MessageListener implements Swift_Events_SendListener, Swift_Events_TransportExceptionListener, LoggerAwareInterface
{
    use LoggerTrait;
    /**
     * @var Swift_Mime_Message
     */
    private $lastMessage;
    /**
     * @var ObjectManager
     */
    private $objectManager;
    /**
     * @var MailAddressStateChanger
     */
    private $mailStateChanger;

    public function __construct(ObjectManager $objectManager, MailAddressStateChanger $mailStateChanger)
    {
        $this->objectManager = $objectManager;
        $this->mailStateChanger = $mailStateChanger;
    }

    /**
     * Invoked immediately before the Message is sent.
     *
     * @param Swift_Events_SendEvent $evt
     */
    public function beforeSendPerformed(Swift_Events_SendEvent $evt)
    {
        $this->lastMessage = $evt->getMessage();
    }

    /**
     * Invoked immediately after the Message is sent.
     *
     * @param Swift_Events_SendEvent $evt
     */
    public function sendPerformed(Swift_Events_SendEvent $evt)
    {
        $message = $evt->getMessage();
        $messageId = $message->getId();
        $emails = [];

        if (null !== $message->getTo()) {
            $emails = array_merge($emails, array_keys($message->getTo()));
        }
        if (null !== $message->getBcc()) {
            $emails = array_merge($emails, array_keys($message->getBcc()));
        }
        if (null !== $message->getCc()) {
            $emails = array_merge($emails, array_keys($message->getCc()));
        }

        switch ($evt->getResult()) {
            case Swift_Events_SendEvent::RESULT_SPOOLED:
            case Swift_Events_SendEvent::RESULT_PENDING:
                $this->mailStateChanger->setPending($messageId, $emails);
                break;
            case Swift_Events_SendEvent::RESULT_FAILED:
                $this->error('email: ' . implode(',', $emails) . '; messageId: ' . $messageId);
                $this->mailStateChanger->setFailed($messageId, $emails);
                break;
            case Swift_Events_SendEvent::RESULT_SUCCESS:
                $this->mailStateChanger->setSent($messageId, $emails);
                break;
            case Swift_Events_SendEvent::RESULT_TENTATIVE:
                $this->mailStateChanger->setSent($messageId, $emails);
                $this->mailStateChanger->setFailed($messageId, array_keys($evt->getFailedRecipients()));
                break;
            default:
        }
        $this->objectManager->flush();
    }

    /**
     * Invoked as a TransportException is thrown in the Transport system.
     *
     * @param Swift_Events_TransportExceptionEvent $evt
     */
    public function exceptionThrown(Swift_Events_TransportExceptionEvent $evt)
    {
        if (null === $this->lastMessage) {
            return;
        }
        $messageId = $this->lastMessage->getId();
        $emails = \array_keys($this->lastMessage->getTo());
        $this->mailStateChanger->setFailed($messageId, $emails);
        $this->objectManager->flush();
        $messageException = $evt->getException()->getMessage();
        $this->error('email: ' . implode(',', $emails) . '; messageId: ' . $messageId . '; message: ' . $messageException . ';');
    }
}
