<?php

declare(strict_types=1);

namespace IssetBV\MailBundle\Command;

use Symfony\Bundle\FrameworkBundle\Command\ContainerAwareCommand;
use Symfony\Component\Console\Input\InputArgument;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;

/**
 * Send Emails from the spool.
 */
class SendSwiftEmailCommand extends ContainerAwareCommand
{
    protected function configure()
    {
        $this
            ->setName('isset_bv:mail:swiftmailer:spool:send')
            ->setDescription('Sends emails from the spool')
            ->addArgument('mailer', InputArgument::REQUIRED)
            ->addOption('message-limit', null, InputOption::VALUE_REQUIRED, 'The maximum number of messages to send.')
            ->addOption('time-limit', null, InputOption::VALUE_REQUIRED, 'The time limit for sending messages (in seconds).')
            ->addOption('recover-timeout', null, InputOption::VALUE_REQUIRED,
                'The timeout for recovering messages that have taken too long to send (in seconds).')
            ->addOption('transport', null, InputOption::VALUE_REQUIRED, 'The service of the transport to use to send the messages.');
    }

    protected function execute(InputInterface $input, OutputInterface $output)
    {
        $this->processMailer($input->getArgument('mailer'), $input, $output);
    }

    private function processMailer($name, InputInterface $input, OutputInterface $output)
    {
        if (!$this->getContainer()->has(sprintf('swiftmailer.mailer.%s', $name))) {
            throw new \InvalidArgumentException(sprintf('The mailer "%s" does not exist.', $name));
        }
        if ($this->getContainer()->getParameter(sprintf('swiftmailer.mailer.%s.spool.enabled', $name))) {
            $mailer = $this->getContainer()->get(sprintf('swiftmailer.mailer.%s', $name));
            $transport = $mailer->getTransport();

            if ($transport instanceof \Swift_Transport_LoadBalancedTransport) {
                foreach ($transport->getTransports() as $eachTransport) {
                    $this->recoverSpool($name, $eachTransport, $input, $output);
                }
            } else {
                $this->recoverSpool($name, $transport, $input, $output);
            }
        } else {
            $output->writeln('<warning>There are no emails to send because the spool is disabled.</warning>');
        }
    }

    private function recoverSpool($name, \Swift_Transport $transport, InputInterface $input, OutputInterface $output)
    {
        if ($transport instanceof \Swift_Transport_SpoolTransport) {
            $spool = $transport->getSpool();
            if ($spool instanceof \Swift_ConfigurableSpool) {
                $spool->setMessageLimit($input->getOption('message-limit'));
                $spool->setTimeLimit($input->getOption('time-limit'));
            }

            if ($spool instanceof \Swift_FileSpool) {
                if (null !== $input->getOption('recover-timeout')) {
                    $spool->recover($input->getOption('recover-timeout'));
                } else {
                    $spool->recover();
                }
            }

            $transportService = $input->getOption('transport') ?: sprintf('swiftmailer.mailer.%s.transport.real', $name);
            $sent = $spool->flushQueue($this->getContainer()->get($transportService));

            $output->writeln(sprintf('%d', $sent));
        }
    }
}
