<?php

declare(strict_types=1);

namespace IssetBV\MailBundle\Service\Mail\Center\Mailer\Swift;

use Exception;
use IssetBV\LogBundle\Log\LoggerTrait;
use IssetBV\MailBundle\Service\Mail\AddressInterface;
use IssetBV\MailBundle\Service\Mail\Center\Mailer\MailChangeableInterface;
use IssetBV\MailBundle\Service\Mail\Center\Mailer\MailerInterface;
use IssetBV\MailBundle\Service\Mail\MailInterface;
use Psr\Log\LoggerAwareInterface;
use Swift_Attachment;
use Swift_Mailer;
use Swift_Message;

class SwiftMailer implements MailerInterface, LoggerAwareInterface
{
    use LoggerTrait;
    /**
     * @var Swift_Mailer
     */
    private $mailer;

    public function __construct(Swift_Mailer $mailer)
    {
        $this->mailer = $mailer;
    }

    public function send(MailChangeableInterface $mailChangeable)
    {
        $mail = $mailChangeable->getMail();

        $message = Swift_Message::newInstance();
        $message->setSubject($mail->getSubject());
        $message->setFrom($mail->getFrom()->getEmail(), $mail->getFrom()->getName());
        $message->setId($mailChangeable->getMessageId());

        $this->addBody($message, $mail);
        $this->AddAttachments($message, $mail);

        $this->addEmails($mailChangeable, $mail->getTo(), function (AddressInterface $address) use ($message) {
            $message->addTo($address->getEmail(), $address->getName());
        });

        $this->addEmails($mailChangeable, $mail->getCc(), function (AddressInterface $address) use ($message) {
            $message->addCc($address->getEmail(), $address->getName());
        });

        $this->addEmails($mailChangeable, $mail->getBcc(), function (AddressInterface $address) use ($message) {
            $message->addBcc($address->getEmail(), $address->getName());
        });

        $this->mailer->send($message);
    }

    /**
     * @param MailChangeableInterface $mailChangeable
     * @param AddressInterface[] $addresses
     * @param callable $add
     */
    private function addEmails(MailChangeableInterface $mailChangeable, array $addresses, callable $add)
    {
        foreach ($addresses as $address) {
            try {
                $add($address);
            } catch (Exception $e) {
                $mailChangeable->changeState($address, MailChangeableInterface::FAILED);
                $this->warning('Address [' . $address->getEmail() . ' / ' . $address->getName() . '] not valid: ' . $e->getMessage());
            }
        }
    }

    private function addBody(Swift_Message $message, MailInterface $mail)
    {
        $html = $mail->getBodyHtml();
        if (null !== $html) {
            $message->setBody($html, 'text/html');
        }
        $text = $mail->getBodyText();
        if (null !== $text) {
            $message->setBody($text);
        }
    }

    private function AddAttachments(Swift_Message $message, MailInterface $mail)
    {
        foreach ($mail->getAttachments() as $attachment) {
            $swiftAttachment = Swift_Attachment::fromPath($attachment->getFilePath());
            $name = $attachment->getName();
            if (null !== $name) {
                $swiftAttachment->setFilename($name);
            }
            $message->attach($swiftAttachment);
        }
    }
}
