<?php
declare(strict_types=1);

namespace IssetBV\MailBundle\Service\Swift;

use Doctrine\Bundle\DoctrineBundle\Registry;
use Doctrine\Common\Persistence\ObjectManager;
use IssetBV\MailBundle\Entity\Mail as MailEntity;
use IssetBV\MailBundle\Entity\MailAddressState;
use IssetBV\MailBundle\Entity\MailAddressType;
use IssetBV\MailBundle\Entity\MailAttachment;
use IssetBV\MailBundle\Service\Mail\Core\Address;
use IssetBV\MailBundle\Service\Mail\Core\Mail;
use IssetBV\MailBundle\Service\Mail\Core\MailCenter as MailCenterInterface;
use IssetBV\MailBundle\Service\Mail\MailEventThrower;
use IssetBV\MailBundle\Service\Mail\MailToEntityConverter;
use Swift_Attachment;
use Swift_Mailer;
use Swift_Message;
use Swift_RfcComplianceException;

/**
 * Class MailCenter
 * @package IssetBV\MailBundle\Service\Swift
 */
class MailCenter implements MailCenterInterface
{

    /**
     * @var Swift_Mailer
     */
    private $mailer;
    /**
     * @var MailToEntityConverter
     */
    private $mailToEntityConverter;
    /**
     * @var ObjectManager
     */
    private $manager;
    /**
     * @var MailEventThrower
     */
    private $mailEventThrower;

    /**
     * MailCenter constructor.
     * @param Registry $registry
     * @param Swift_Mailer $mailer
     * @param MailToEntityConverter $mailToEntityConverter
     * @param MailEventThrower $mailEventThrower
     */
    public function __construct(
        Registry $registry,
        Swift_Mailer $mailer,
        MailToEntityConverter $mailToEntityConverter,
        MailEventThrower $mailEventThrower
    ) {
        $this->mailer = $mailer;
        $this->manager = $registry->getManagerForClass(MailEntity::class);
        $this->mailToEntityConverter = $mailToEntityConverter;
        $this->mailEventThrower = $mailEventThrower;
    }

    /**
     * @param Mail $mail
     * @return MailEntity
     */
    public function send(Mail $mail): MailEntity
    {
        $message = Swift_Message::newInstance();
        $message->setSubject($mail->getSubject());
        $message->setFrom($mail->getFrom()->getEmail(), $mail->getFrom()->getName());

        $html = $mail->getBodyHtml();
        if ($html !== null) {
            $message->setBody($html, 'text/html');
        }
        $text = $mail->getBodyText();
        if ($text !== null) {
            $message->setBody($text);
        }

        $failedState = $this->manager->getRepository(MailAddressState::class)->find(MailAddressState::FAILED);
        $mailEntity = $this->mailToEntityConverter->convert($mail);
        $mailEntity->setMessageId($message->getId());
        $this->manager->persist($mailEntity);
        foreach ($mail->getAttachments() as $attachment) {
            $swiftAttachment = Swift_Attachment::fromPath($attachment->getFilePath());
            $mailAttachment = new MailAttachment();
            $mailAttachment->setMail($mailEntity);
            $mailAttachment->setFile($attachment->getFilePath());
            $name = $attachment->getName();
            if ($name !== null) {
                $swiftAttachment->setFilename($name);
                $mailAttachment->setName($name);
            }
            $message->attach($swiftAttachment);
            $this->manager->persist($mailAttachment);
        }

        foreach ($mail->getTo() as $data) {
            try {
                $message->addTo($data->getEmail(), $data->getName());
            } catch (Swift_RfcComplianceException $e) {
                $mailEntity->setStateMailAddress($data->getEmail(), MailAddressType::TO, $failedState);
            }
        }

        foreach ($mail->getCc() as $data) {
            try {
                $message->addCc($data->getEmail(), $data->getName());
            } catch (Swift_RfcComplianceException $e) {
                $mailEntity->setStateMailAddress($data->getEmail(), MailAddressType::CC, $failedState);
            }
        }

        foreach ($mail->getBcc() as $data) {
            try {
                $message->addBcc($data->getEmail(), $data->getName());
            } catch (Swift_RfcComplianceException $e) {
                $mailEntity->setStateMailAddress($data->getEmail(), MailAddressType::BCC, $failedState);
            }
        }

        $this->manager->flush();
        $this->mailer->send($message);
        $this->mailEventThrower->throwSpoolEvent($mailEntity);
        return $mailEntity;
    }

}