<?php
declare(strict_types = 1);

namespace IssetBV\MailBundle\Service\Swift;

use Doctrine\ORM\EntityManagerInterface;
use IssetBV\MailBundle\Service\Mail\MailStateChanger;
use Psr\Log\LoggerInterface;
use Swift_Events_SendEvent;
use Swift_Events_SendListener;
use Swift_Events_TransportExceptionEvent;
use Swift_Events_TransportExceptionListener;
use Swift_Mime_Message;

/**
 * Class MessageListener
 * @package IssetBV\MailBundle\Service\Swift
 */
class MessageListener implements Swift_Events_SendListener, Swift_Events_TransportExceptionListener
{

    /**
     * @var EntityManagerInterface
     */
    private $entityManager;

    /**
     * @var LoggerInterface
     */
    private $logger;

    /**
     * @var Swift_Mime_Message
     */
    private $lastMessage;
    /**
     * @var MailStateChanger
     */
    private $mailStateChanger;

    /**
     * MessageListener constructor.
     * @param EntityManagerInterface $entityManager
     * @param MailStateChanger $mailStateChanger
     * @param LoggerInterface $logger
     */
    public function __construct(EntityManagerInterface $entityManager, MailStateChanger $mailStateChanger, LoggerInterface $logger)
    {
        $this->entityManager = $entityManager;
        $this->logger = $logger;
        $this->mailStateChanger = $mailStateChanger;
    }

    /**
     * Invoked immediately before the Message is sent.
     *
     * @param Swift_Events_SendEvent $evt
     */
    public function beforeSendPerformed(Swift_Events_SendEvent $evt)
    {
        $this->lastMessage = $evt->getMessage();
    }

    /**
     * Invoked immediately after the Message is sent.
     *
     * @param Swift_Events_SendEvent $evt
     */
    public function sendPerformed(Swift_Events_SendEvent $evt)
    {
        $message = $evt->getMessage();
        $messageId = $message->getId();
        $emails = array_merge(array_keys($message->getTo()), array_keys($message->getBcc()), array_keys($message->getCc()));
        switch ($evt->getResult()) {
            case Swift_Events_SendEvent::RESULT_SPOOLED:
            case Swift_Events_SendEvent::RESULT_PENDING:
                $this->mailStateChanger->setPending($messageId, $emails);
                break;
            case Swift_Events_SendEvent::RESULT_FAILED:
                $this->logger->error('email: ' . implode(',', $emails) . '; messageId: ' . $messageId);
                $this->mailStateChanger->setFailed($messageId, $emails);
                break;
            case Swift_Events_SendEvent::RESULT_SUCCESS:
                $this->mailStateChanger->setSent($messageId, $emails);
                break;
            case Swift_Events_SendEvent::RESULT_TENTATIVE:
                $this->mailStateChanger->setSent($messageId, $emails);
                $this->mailStateChanger->setFailed($messageId, array_keys($evt->getFailedRecipients()));
                break;
        }
        $this->entityManager->flush();
    }

    /**
     * Invoked as a TransportException is thrown in the Transport system.
     *
     * @param Swift_Events_TransportExceptionEvent $evt
     */
    public function exceptionThrown(Swift_Events_TransportExceptionEvent $evt)
    {
        if ($this->lastMessage === null) {
            return;
        }
        $messageId = $this->lastMessage->getId();
        $emails = array_keys($this->lastMessage->getTo());
        $this->mailStateChanger->setFailed($messageId, $emails);
        $this->entityManager->flush();
        $messageException = $evt->getException()->getMessage();
        $this->logger->error('email: ' . implode(',', $emails) . '; messageId: ' . $messageId . '; message: ' . $messageException . ';');
    }

}