<?php
declare(strict_types = 1);

namespace IssetBV\MailBundle\Service\Swift;

use Doctrine\Bundle\DoctrineBundle\Registry;
use Doctrine\Common\Persistence\ObjectManager;
use IssetBV\MailBundle\Entity\Mail as MailEntity;
use IssetBV\MailBundle\Entity\MailAttachment;
use IssetBV\MailBundle\Service\Mail\Core\Address;
use IssetBV\MailBundle\Service\Mail\Core\Mail;
use IssetBV\MailBundle\Service\Mail\Core\MailCenter as MailCenterInterface;
use IssetBV\MailBundle\Service\Mail\MailToEntityConverter;
use Swift_Attachment;
use Swift_Mailer;
use Swift_Message;

/**
 * Class MailCenter
 * @package IssetBV\MailBundle\Service\Swift
 */
class MailCenter implements MailCenterInterface
{

    /**
     * @var Swift_Mailer
     */
    private $mailer;
    /**
     * @var MailToEntityConverter
     */
    private $mailToEntityConverter;
    /**
     * @var ObjectManager
     */
    private $manager;

    /**
     * MailCenter constructor.
     * @param Registry $registry
     * @param Swift_Mailer $mailer
     * @param MailToEntityConverter $mailToEntityConverter
     */
    public function __construct(Registry $registry, Swift_Mailer $mailer, MailToEntityConverter $mailToEntityConverter)
    {
        $this->mailer = $mailer;
        $this->manager = $registry->getManagerForClass(MailEntity::class);
        $this->mailToEntityConverter = $mailToEntityConverter;
    }

    /**
     * @param Mail $mail
     * @return MailEntity
     */
    public function send(Mail $mail): MailEntity
    {
        $message = Swift_Message::newInstance();
        $message->setSubject($mail->getSubject());
        $message->setFrom($mail->getFrom()->getEmail(), $mail->getFrom()->getName());

        $message->setTo($this->convertAddressesToSwift($mail->getTo()));
        $message->setCc($this->convertAddressesToSwift($mail->getCc()));
        $message->setBcc($this->convertAddressesToSwift($mail->getBcc()));

        $html = $mail->getBodyHtml();
        if ($html !== null) {
            $message->setBody($html, 'text/html');
        }
        $text = $mail->getBodyText();
        if ($text !== null) {
            $message->setBody($text);
        }
        $mailEntity = $this->mailToEntityConverter->convert($mail);
        $mailEntity->setMessageId($message->getId());
        $this->manager->persist($mailEntity);
        foreach ($mail->getAttachments() as $attachment) {
            $swiftAttachment = Swift_Attachment::fromPath($attachment->getFilePath());
            $mailAttachment = new MailAttachment();
            $mailAttachment->setMail($mailEntity);
            $mailAttachment->setFile($attachment->getFilePath());
            $name = $attachment->getName();
            if ($name !== null) {
                $swiftAttachment->setFilename($name);
                $mailAttachment->setName($name);
            }
            $message->attach($swiftAttachment);
            $this->manager->persist($mailAttachment);
        }
        $this->manager->flush();
        $this->mailer->send($message);
        return $mailEntity;
    }

    /**
     * @param Address[] $addresses
     * @return array
     */
    private function convertAddressesToSwift(array $addresses): array
    {
        $return = [];
        foreach ($addresses as $address) {
            $name = $address->getName();
            if ($name === null) {
                $return[] = $address->getEmail();
            } else {
                $return[$address->getEmail()] = $name;
            }
        }

        return $return;

    }

}