<?php

declare(strict_types=1);

namespace IssetBV\LogBundle\Log\Processor;

use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\RequestStack;

class RouteLogProcessor
{
    /**
     * @var RequestStack|null
     */
    private $requestStack;
    /**
     * @var array
     */
    private $config;

    public function __construct(array $config)
    {
        $this->config = $config;
    }

    public function processRecord(array $record): array
    {
        $request = $this->getRequest();
        if (null === $request) {
            return $record;
        }

        if ($this->config['route']) {
            $record['extra']['route'] = $this->getRoute($request) ?? '';
        }
        if ($this->config['route_path']) {
            $record['extra']['route_path'] = $this->getPath($request) ?? '';
        }
        if ($this->config['headers']) {
            $record['extra']['headers'] = json_encode($request->headers->all());
        }
        if ($this->config['request']) {
            $record['extra']['request'] = json_encode($request->request->all());
        }
        if ($this->config['query']) {
            $record['extra']['query'] = json_encode($request->query->all());
        }
        if ($this->config['cookies']) {
            $record['extra']['cookies'] = json_encode($request->cookies->all());
        }

        return $record;
    }

    public function setRequest(RequestStack $requestStack)
    {
        $this->requestStack = $requestStack;
    }

    /**
     * @param Request $request
     *
     * @return string|null
     */
    private function getRoute(Request $request)
    {
        return $request->get('_route');
    }

    /**
     * @param Request $request
     *
     * @return string|null
     */
    private function getPath(Request $request)
    {
        return str_replace($request->getBasePath(), '', $request->getRequestUri());
    }

    /**
     * @return null|Request
     */
    private function getRequest()
    {
        if (null === $this->requestStack) {
            return null;
        }

        $request = $this->requestStack->getCurrentRequest();
        if (null !== $request) {
            return $request;
        }
    }
}
