(function () {
    'use strict';

    var jobBuilderService = function ($rootScope, $http) {
        var activeJob = {};
        var self = this;
        var templates = {};
        var presets = {};
        var remotes = {};

        this.initRemotes = function () {
            $http.get(Routing.generate('isset_bv_hatchery_application_job_builder_remotes'))
                .then(function (result) {
                    remotes = result.data;
                    $rootScope.$broadcast('remotes:changed');
                });
        };
        this.initPresets = function () {
            $http.get(Routing.generate('isset_bv_hatchery_application_job_builder_presets', {actionType: 'video-transcode'}))
                .then(function (result) {
                    presets = result.data;
                    $rootScope.$broadcast('presets:changed');
                });
        };
        this.initTemplates = function () {
            $http.get(Routing.generate('isset_bv_hatchery_application_job_builder_templates'))
                .then(function (result) {
                    templates = result.data;
                    $rootScope.$broadcast('templates:changed');
                });
        };

        this.loadTemplate = function (templateId) {
            return $http.get(Routing.generate('isset_bv_hatchery_application_job_builder_templates_data', {id: templateId}));
        };

        this.init = function () {
            self.initRemotes();
            self.initPresets();
            self.initTemplates();
        };

        this.getRemotes = function () {
            return remotes;
        };
        this.getPresets = function () {
            return presets;
        };
        this.getTemplates = function () {
            return templates;
        };

        this.setActiveJob = function (job) {
            activeJob = job;
            $rootScope.$broadcast('job:updated');
        };

        this.getActiveJob = function () {
            if (false === angular.isArray(activeJob.inputs)) {
                activeJob.inputs = [];
            }
            return activeJob;
        };

        this.updateApiJob = function (job) {
            var request = {};

            request.inputs = [];
            request.outputs = [];

            var playlist = {};
            playlist.type = 'playlist';
            playlist.reference = self.generateUuid();
            playlist.filename = 'playlist.m3u8';
            playlist.streams = [];

            if (job.strict === false) {
                request.strict = false;
            }

            angular.forEach(job.inputs, function (inputItem) {
                var input = {};
                input.reference = inputItem.id;
                input.url = self.buildUrl(inputItem, true);
                request.inputs.push(input);

                angular.forEach(inputItem.outputs, function (outputItem) {
                    var output = {};
                    output.reference = outputItem.id;
                    output.source = input.reference;
                    output.filename = outputItem.filename;

                    if (outputItem.remote && outputItem.remote.type === 'download') {
                        output.download = true;
                    } else if (!outputItem.temporaryTranscode) {
                        output.base_url = self.buildBaseUrl(outputItem);
                    }

                    request.outputs.push(output);
                    self.buildOptions(output, outputItem, ['preset', 'seek_offset', 'output_length', 'width', 'height', 'deinterlace', 'type']);


                    if (outputItem.type === 'segmented') {
                        playlist.base_url = output.base_url;
                        playlist.streams.push(output.reference);
                    }

                    angular.forEach(outputItem.tasks, function (taskItem) {

                        if (taskItem === 'caption') {
                            output.caption_url = self.buildUrl(taskItem);
                        } else if (taskItem._type === 'transfer') {
                            var transfer = {};
                            transfer.type = "transfer-only";
                            transfer.source = output.reference;
                            transfer.base_url = self.buildBaseUrl(taskItem);
                            transfer.filename = taskItem.filename;
                            request.outputs.push(transfer);
                        } else if (taskItem._type === 'watermark') {
                            if (!output.watermarks) {
                                output.watermarks = [];
                            }

                            var watermark = {};
                            watermark.url = self.buildUrl(taskItem);
                            self.buildOptions(watermark, taskItem, ['width', 'height', 'x', 'y']);

                            output.watermarks.push(watermark);
                        } else if (taskItem._type === 'stills') {
                            if (!output.stills) {
                                output.stills = [];
                            }

                            var stills = {};
                            stills.base_url = self.buildBaseUrl(taskItem);
                            stills.filename = taskItem.filename;
                            self.buildOptions(stills, taskItem, ['width', 'height', 'amount', 'format', 'timestamp']);

                            output.stills.push(stills);
                        }
                    });
                });
            });

            angular.forEach(job.concatenates, function (concatenateItem) {
                var output = {};
                output.type = 'concatenate';
                output.sources = concatenateItem.sources;
                output.base_url = self.buildBaseUrl(concatenateItem);
                output.filename = concatenateItem.filename;

                self.buildOptions(output, concatenateItem, ['preset', 'seek_offset', 'output_length', 'width', 'height', 'deinterlace', 'type']);
                request.outputs.push(output);
            });

            if (playlist.streams.length > 0) {
                request.outputs.push(playlist);
            }

            return request;

        };


        this.buildUrl = function (element) {

            if (element.remote && element.remote.id === 'upload') {
                if (element.file && element.file.location) {
                    return element.file.location;
                } else if (element.file && !element.file.location) {
                    return '';
                }
            }

            if (element.remote && element.remote.id === 'archive') {
                if (element.file) {
                    return 'va://my.videoarchive.io/' + element.file.location;
                } else if (element.remote.data &&  element.remote.data.directory) {
                    return 'va://my.videoarchive.io/' + element.remote.data.directory;
                } else {
                    return 'va://my.videoarchive.io/';
                }
            }

            if (element.remote && element.file) {
                return 'remote://' + element.remote.id + '/' + this.encodeAnything(element.file.location);
            } else if (element.remote) {
                return 'remote://' + element.remote.id;
            } else {
                return '';
            }
        };

        this.buildBaseUrl = function (element) {

            if (element.remote && element.remote.id === 'archive') {
                if (element.remote.data && element.remote.data.directory) {
                    return 'va://my.videoarchive.io/' + element.remote.data.directory;
                } else {
                    return 'va://my.videoarchive.io/';
                }
            }

            if (element.remote && element.remote.data && element.remote.data.directory) {
                return 'remote://' + element.remote.id + '/' + this.encodeAnything(element.remote.data.directory);
            } else if (element.remote) {
                return 'remote://' + element.remote.id + '/';
            } else {
                return '';
            }
        };

        this.buildOptions = function (output, element, options) {
            angular.forEach(options, function (option) {
                if (element.hasOwnProperty(option) && element[option]) {
                    if (angular.isObject(element[option]) && 'name' in element[option]) {
                        output[option] = element[option].name;
                    } else {
                        output[option] = element[option];
                    }
                }
            });
        };

        /**
         * Generate random uuid
         * @returns {string}
         */
        this.generateUuid = function () {
            function s4() {
                return Math.floor((1 + Math.random()) * 0x10000)
                    .toString(16)
                    .substring(1);
            }

            return s4() + s4() + '-' + s4() + '-' + s4() + '-' +
                s4() + '-' + s4() + s4() + s4();
        };

        /**
         * Convert seconds to timestamp
         * @param secondsRaw
         * @returns {string}
         */
        this.secondsToTimestamp = function (secondsRaw) {
            var sec_num = parseInt(secondsRaw, 10);

            var hours = Math.floor(sec_num / 3600);
            var minutes = Math.floor((sec_num - (hours * 3600)) / 60);
            var seconds = sec_num - (hours * 3600) - (minutes * 60);

            if (hours < 10) {
                hours = "0" + hours;
            }
            if (minutes < 10) {
                minutes = "0" + minutes;
            }
            if (seconds < 10) {
                seconds = "0" + seconds;
            }

            return hours + ':' + minutes + ':' + seconds;
        };

        /**
         * Encode anything
         * @param anything
         * @returns {string}
         */
        this.encodeAnything = function (anything) {

            var parts = anything.split('/');
            var encodedParts = [];

            angular.forEach(parts, function (part) {
                encodedParts.push(encodeURIComponent(part));
            });

            return encodedParts.join('/');
        };
    };


    angular
        .module('jobBuilder')
        .service('jobBuilderService', [
            '$rootScope',
            '$http',
            jobBuilderService
        ]);

})();