<?php

declare(strict_types=1);

namespace IssetBV\Http\Guzzle;

use Exception;
use GuzzleHttp\Client;
use GuzzleHttp\Psr7\Request;
use GuzzleHttp\RequestOptions;
use IssetBV\Http\Client\ClientInterface;
use IssetBV\Http\Client\OptionsInterface;
use IssetBV\Http\Client\RequestGeneratorInterface;
use IssetBV\Http\Client\StreamGeneratorInterface;
use Psr\Http\Message\RequestInterface;
use Psr\Http\Message\ResponseInterface;
use Psr\Http\Message\StreamInterface;

class GuzzleClient implements ClientInterface
{
    /**
     * @var Client|null
     */
    private $client;
    /**
     * @var GuzzleRequestGenerator|null
     */
    private $requestGenerator;
    /**
     * @var GuzzleStreamGenerator|null
     */
    private $streamGenerator;

    /**
     * @return Client
     */
    public function getClient(): Client
    {
        if (null === $this->client) {
            $this->client = new Client([
                'http_errors' => false,
            ]);
        }

        return $this->client;
    }

    /**
     * @param RequestInterface $request
     * @param OptionsInterface|null $options
     *
     * @throws GuzzleConnectException
     *
     * @return ResponseInterface
     */
    public function call(RequestInterface $request, OptionsInterface $options = null): ResponseInterface
    {
        try {
            $clientOptions = $this->createGuzzleOptionsFromOptions($options);

            return $this->getClient()->send($request, $clientOptions);
        } catch (Exception $guzzleException) {
            throw new GuzzleConnectException($guzzleException->getMessage(), $guzzleException->getCode(), $guzzleException);
        }
    }

    /**
     * @param string $method
     * @param string $uri
     * @param array $headers
     * @param string|null|resource|StreamInterface $body
     * @param string $version
     *
     * @return RequestInterface
     */
    public function create(
        string $method,
        string $uri,
        array $headers = [],
        $body = null,
        string $version = '1.1'
    ): RequestInterface {
        return new Request($method, $uri, $headers, $body, $version);
    }

    /**
     * @deprecated
     *
     * @return RequestGeneratorInterface
     */
    public function getRequestGenerator(): RequestGeneratorInterface
    {
        if (null === $this->requestGenerator) {
            $this->requestGenerator = new GuzzleRequestGenerator();
        }

        return $this->requestGenerator;
    }

    /**
     * @return StreamGeneratorInterface
     */
    public function getStreamGenerator(): StreamGeneratorInterface
    {
        if (null === $this->streamGenerator) {
            $this->streamGenerator = new GuzzleStreamGenerator();
        }

        return $this->streamGenerator;
    }

    private function createGuzzleOptionsFromOptions(OptionsInterface $options = null): array
    {
        $guzzleOptions = [];
        if (null === $options) {
            return $guzzleOptions;
        }

        foreach ($options->getOptions() as $optionName => $optionValue) {
            switch ($optionName) {
                case OptionsInterface::CONNECT_TIMEOUT:
                    $guzzleOptions[RequestOptions::CONNECT_TIMEOUT] = $optionValue;
                    break;
                case OptionsInterface::READ_TIMEOUT:
                    $guzzleOptions[RequestOptions::READ_TIMEOUT] = $optionValue;
                    break;
            }
        }

        return $guzzleOptions;
    }
}
