<?php

declare(strict_types=1);

namespace IssetBV\HttpCallbackDoctrineBundle\Service;

use DateTime;
use Doctrine\Common\Persistence\ObjectManager;
use IssetBV\Http\Client\ClientInterface;
use IssetBV\Http\Client\Exception\ConnectExceptionInterface;
use IssetBV\Http\Client\Options;
use IssetBV\Http\Client\OptionsInterface;
use IssetBV\HttpCallback\HttpCallbackTryHandlerInterface;
use IssetBV\HttpCallback\HttpCallbackTryInterface;
use IssetBV\HttpCallback\Processor\ProcessorInterface;
use IssetBV\HttpCallbackDoctrineBundle\Entity\HttpCallbackTry;
use RuntimeException;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;

class HttpCallbackTryHandler implements HttpCallbackTryHandlerInterface
{
    /**
     * @var ClientInterface
     */
    private $client;
    /**
     * @var ObjectManager
     */
    private $objectManager;
    /**
     * @var EventDispatcherInterface
     */
    private $eventDispatcher;

    public function __construct(ObjectManager $objectManager, ClientInterface $client, EventDispatcherInterface $eventDispatcher)
    {
        $this->client = $client;
        $this->objectManager = $objectManager;
        $this->eventDispatcher = $eventDispatcher;
    }

    public function handle(HttpCallbackTryInterface $httpCallbackTry)
    {
        if (!($httpCallbackTry instanceof HttpCallbackTry)) {
            throw new RuntimeException('HttpCallbackTryHandler: expected <' . HttpCallbackTry::class . '> got <' . \get_class($httpCallbackTry) . '>');
        }

        $httpCallback = $httpCallbackTry->getHttpCallbackDataInterface();

        $body = $httpCallback->getBody();
        $headers = $httpCallback->getHeaders();
        if (\is_array($body)) {
            if (!array_key_exists('Content-Type', $headers)) {
                $headers['Content-Type'] = 'application/x-www-form-urlencoded';
            }
            $body = http_build_query($body);
        }

        $request = $this->client->create(
            $httpCallback->getMethod(),
            $httpCallback->getUrl(),
            $headers,
            $body,
            $httpCallback->getVersion()
        );

        $triesCount = $httpCallbackTry->getHttpCallback()->getTries()->count();

        $options = new Options();
        $options->setOption(OptionsInterface::CONNECT_TIMEOUT, $triesCount);
        $options->setOption(OptionsInterface::READ_TIMEOUT, $triesCount + 1);

        try {
            $response = $this->client->call($request, $options);
            $httpCallbackTry->setStatusCode($response->getStatusCode());
            $httpCallbackTry->setResponse((string) $response->getBody());

            $responseValidator = $httpCallback->getResponseValidator();
            if (null === $responseValidator || $responseValidator->validateResponse($response)) {
                $this->handleSuccess($httpCallbackTry);
            } else {
                $this->handleFailed($httpCallbackTry);
            }
        } catch (ConnectExceptionInterface $connectException) {
            $httpCallbackTry->setResponse((string) $connectException->getMessage());
            $this->handleFailed($httpCallbackTry);
        }
    }

    public function handleSuccess(HttpCallbackTry $callbackTry)
    {
        $callbackTry->setStatus(ProcessorInterface::STATUS_SUCCESS);
        $callbackTry->getHttpCallback()->setSentDate(new DateTime());
        $this->objectManager->flush();
    }

    private function handleFailed(HttpCallbackTry $callbackTry)
    {
        $callbackTry->setStatus(ProcessorInterface::STATUS_FAILED);

        if ($callbackTry->getHttpCallback()->reTry()) {
            $this->handleNewReTry($callbackTry);
        } else {
            $callbackTry->getHttpCallback()->setFailed();
            $this->objectManager->flush();
        }
    }

    private function handleNewReTry(HttpCallbackTry $callbackTry)
    {
        $countTries = $callbackTry->getHttpCallback()->getTries()->count();

        switch ($countTries) {
            case 1:
                $dateSend = new DateTime('+5 minutes');
                break;
            case 2:
                $dateSend = new DateTime('+30 minutes');
                break;
            case 3:
                $dateSend = new DateTime('+60 minutes');
                break;
            default:
                $dateSend = new DateTime('+1 day');
                break;
        }

        $callbackTry = new HttpCallbackTry(
            ProcessorInterface::TYPE_RETRY,
            ProcessorInterface::STATUS_PENDING,
            $callbackTry->getHttpCallback(),
            $dateSend
        );

        $this->objectManager->persist($callbackTry);
        $this->objectManager->flush();

        $httpCallbackAddedEvent = new HttpCallbackTryEvent($callbackTry);
        $this->eventDispatcher->dispatch(HttpCallbackTryEvent::EVENT_NAME, $httpCallbackAddedEvent);
    }
}
