<?php

declare(strict_types=1);

namespace IssetBV\HttpCallbackDoctrineBundle\Entity;

use DateTime;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\Mapping as ORM;
use IssetBV\HttpCallback\HttpCallbackDataInterface;
use IssetBV\HttpCallback\Response\ResponseValidatorInterface;
use Ramsey\Uuid\Uuid;

/**
 * @ORM\Table(name="issetbv_http_callbacks")
 * @ORM\Entity(repositoryClass="IssetBV\HttpCallbackDoctrineBundle\Repository\HttpCallbackRepository")
 */
class HttpCallback implements HttpCallbackDataInterface
{
    /**
     * @var int|null
     *
     * @ORM\Column(name="id", type="integer", nullable=false)
     * @ORM\Id
     * @ORM\GeneratedValue(strategy="IDENTITY")
     */
    private $id;
    /**
     * @var int
     *
     * @ORM\Column(type="integer", nullable=false)
     */
    private $maxTries = 5;
    /**
     * @var HttpCallbackTry[]|ArrayCollection
     *
     * @ORM\OneToMany(targetEntity="HttpCallbackTry", mappedBy="httpCallback")
     */
    private $tries;
    /**
     * @var string|null
     *
     * @ORM\Column(type="string", nullable=true)
     */
    private $responseValidator;
    /**
     * @var string
     *
     * @ORM\Column(type="string", nullable=false)
     */
    private $method;
    /**
     * @var string
     *
     * @ORM\Column(type="text", nullable=false)
     */
    private $url;
    /**
     * @var array
     *
     * @ORM\Column(type="json_array", nullable=false)
     */
    private $headers;
    /**
     * @var string
     *
     * @ORM\Column(type="string", nullable=false, unique=true)
     */
    private $uuid;
    /**
     * @var string|null
     *
     * @ORM\Column(type="text", nullable=true)
     */
    private $body;
    /**
     * @var string
     *
     * @ORM\Column(type="string", nullable=false)
     */
    private $version;
    /**
     * @var string|null
     *
     * @ORM\Column(type="string", nullable=true)
     */
    private $groupIdentifier;
    /**
     * @var DateTime|null
     *
     * @ORM\Column(name="date_sent", type="datetime", nullable=true)
     */
    private $sentDate;
    /**
     * @var DateTime|null
     *
     * @ORM\Column(name="date_failed", type="datetime", nullable=true)
     */
    private $failedDate;

    /**
     * @var DateTime|null
     *
     * @ORM\Column(name="date_cleaned", type="datetime", nullable=true)
     */
    private $cleanedDate;
    /**
     * @var DateTime
     *
     * @ORM\Column(name="date_created", type="datetime", nullable=false)
     */
    private $created;

    public function __construct()
    {
        $this->tries = new ArrayCollection();
        $this->created = new DateTime();
        $this->uuid = Uuid::uuid4()->toString();
    }

    /**
     * @return int
     */
    public function getMaxTries(): int
    {
        return $this->maxTries;
    }

    /**
     * @return ArrayCollection|HttpCallbackTry[]
     */
    public function getTries()
    {
        return $this->tries;
    }

    /**
     * @return string
     */
    public function getMethod(): string
    {
        return $this->method;
    }

    /**
     * @param string $method
     */
    public function setMethod(string $method)
    {
        $this->method = $method;
    }

    /**
     * @return string
     */
    public function getUrl(): string
    {
        return $this->url;
    }

    /**
     * @param string $url
     */
    public function setUrl(string $url)
    {
        $this->url = $url;
    }

    /**
     * @return array
     */
    public function getHeaders(): array
    {
        return $this->headers;
    }

    /**
     * @param array $headers
     */
    public function setHeaders(array $headers)
    {
        $this->headers = $headers;
    }

    /**
     * @return null|string
     */
    public function getBody()
    {
        return $this->body;
    }

    /**
     * @param null|string $body
     */
    public function setBody(string $body = null)
    {
        $this->body = $body;
    }

    /**
     * @return string
     */
    public function getVersion(): string
    {
        return $this->version;
    }

    /**
     * @param string $version
     */
    public function setVersion(string $version)
    {
        $this->version = $version;
    }

    /**
     * @return null|string
     */
    public function getGroupIdentifier()
    {
        return $this->groupIdentifier;
    }

    /**
     * @param null|string $groupIdentifier
     */
    public function setGroupIdentifier(string $groupIdentifier = null)
    {
        $this->groupIdentifier = $groupIdentifier;
    }

    public function setResponseValidator(ResponseValidatorInterface $responseValidator = null)
    {
        $this->responseValidator = null !== $responseValidator ? serialize($responseValidator) : null;
    }

    /**
     * @return ResponseValidatorInterface|null
     */
    public function getResponseValidator()
    {
        if (null === $this->responseValidator) {
            return null;
        }
        $class = unserialize($this->responseValidator);

        return $class instanceof ResponseValidatorInterface ? $class : null;
    }

    public function getUUID(): string
    {
        return $this->uuid;
    }

    public function setSentDate(DateTime $sentDate)
    {
        $this->sentDate = $sentDate;
    }

    public function reTry(): bool
    {
        $maxTries = $this->getMaxTries();
        if (0 === $maxTries) {
            return false;
        }

        return $maxTries > $this->getTries()->count();
    }

    public function addTry(HttpCallbackTry $httpCallbackTry)
    {
        $this->tries[] = $httpCallbackTry;
    }

    public function setFailed()
    {
        $this->failedDate = new DateTime();
    }

    public function setCleaned()
    {
        $this->cleanedDate = new DateTime();
    }
}
