<?php

declare(strict_types=1);

namespace IssetBV\FormBundle\Service\Error;

use JsonSerializable;

/**
 * Class FormData.
 */
class FormData implements JsonSerializable
{
    /**
     * @var bool
     */
    private $valid;
    /**
     * @var string
     */
    private $path;
    /**
     * @var FormData|null
     */
    private $parent;
    /**
     * @var FormDataError[]
     */
    private $errors = [];
    /**
     * @var FormData[]
     */
    private $fields = [];

    public function __construct(bool $valid, string $path, self $parent = null)
    {
        $this->path = $path;
        $this->parent = $parent;
        $this->setValid($valid);
    }

    /**
     * @param string $path
     */
    public function setPath(string $path)
    {
        $this->path = $path;
    }

    /**
     * @return bool
     */
    public function isValid(): bool
    {
        return $this->valid;
    }

    /**
     * @return string
     */
    public function getPath(): string
    {
        return $this->path;
    }

    /**
     * @param FormDataError $formDataError
     */
    public function addError(FormDataError $formDataError)
    {
        $this->setValid(false);
        $this->errors[] = $formDataError;
    }

    /**
     * @return FormDataError[]
     */
    public function getErrors(): array
    {
        return $this->errors;
    }

    /**
     * @return array
     */
    public function getErrorsArray(): array
    {
        $return = [];
        foreach ($this->errors as $name => $error) {
            $return[$name] = $error->toArray();
        }

        return $return;
    }

    /**
     * @param mixed $field
     * @param FormData $formData
     */
    public function addField($field, self $formData)
    {
        $this->fields[$field] = $formData;
        if (!$formData->isValid()) {
            $this->setValid($formData->isValid());
        }
    }

    /**
     * @return FormData[]
     */
    public function getFields(): array
    {
        return $this->fields;
    }

    public function getFieldAsArray()
    {
        $return = [];
        foreach ($this->fields as $name => $field) {
            $return[$name] = $field->toArray();
        }

        return $return;
    }

    public function toArray(): array
    {
        return [
            'valid' => $this->isValid(),
            'path' => $this->getPath(),
            'errors' => $this->getErrorsArray(),
            'fields' => $this->getFieldAsArray(),
        ];
    }

    public function setValid(bool $valid)
    {
        $this->valid = $valid;
        if (!$this->valid && null !== $this->parent) {
            $this->parent->setValid(false);
        }
    }

    /**
     * @return FormData|null
     */
    public function getParent()
    {
        return $this->parent;
    }

    public function hasField($field): bool
    {
        return array_key_exists($field, $this->fields);
    }

    public function getField($field): self
    {
        if (!$this->hasField($field)) {
            throw new \LogicException('Field not found: ' . $field);
        }

        return $this->fields[$field];
    }

    public function jsonSerialize()
    {
        return $this->toArray();
    }
}
