<?php

declare(strict_types=1);

namespace IssetBV\FormBundle\Service\Error;

use JsonSerializable;

/**
 * Class FormData.
 */
class FormData implements JsonSerializable
{
    /**
     * @var bool
     */
    private $valid;
    /**
     * @var string
     */
    private $path;
    /**
     * @var FormDataError[]
     */
    private $errors = [];
    /**
     * @var FormData[]
     */
    private $fields = [];

    /**
     * FormData constructor.
     *
     * @param bool $valid
     * @param string $path
     */
    public function __construct(bool $valid, string $path)
    {
        $this->valid = $valid;
        $this->path = $path;
    }

    /**
     * @return bool
     */
    public function isValid(): bool
    {
        return $this->valid;
    }

    /**
     * @return string
     */
    public function getPath(): string
    {
        return $this->path;
    }

    /**
     * @param FormDataError $formDataError
     */
    public function addError(FormDataError $formDataError)
    {
        $this->errors[] = $formDataError;
    }

    /**
     * @return FormDataError[]
     */
    public function getErrors(): array
    {
        return $this->errors;
    }

    /**
     * @return array
     */
    public function getErrorsArray(): array
    {
        $return = [];
        foreach ($this->errors as $name => $error) {
            $return[$name] = $error->toArray();
        }

        return $return;
    }

    /**
     * @param mixed $field
     * @param FormData $formData
     */
    public function addField($field, self $formData)
    {
        $this->fields[$field] = $formData;
    }

    /**
     * @return FormData[]
     */
    public function getFields(): array
    {
        return $this->fields;
    }

    public function getFieldAsArray()
    {
        $return = [];
        foreach ($this->fields as $name => $field) {
            $return[$name] = $field->toArray();
        }

        return $return;
    }

    public function toArray(): array
    {
        return [
            'valid' => $this->isValid(),
            'path' => $this->getPath(),
            'errors' => $this->getErrorsArray(),
            'fields' => $this->getFieldAsArray(),
        ];
    }

    public function setValid(bool $valid)
    {
        $this->valid = $valid;
    }

    /**
     * Specify data which should be serialized to JSON.
     *
     * @see http://php.net/manual/en/jsonserializable.jsonserialize.php
     *
     * @return mixed data which can be serialized by <b>json_encode</b>,
     * which is a value of any type other than a resource
     *
     * @since 5.4.0
     */
    public function jsonSerialize()
    {
        return $this->toArray();
    }
}
