<?php

declare(strict_types=1);

namespace IssetBV\FormBundle\Service\Error;

use JsonSerializable;

/**
 * Class FormData.
 */
class FormData implements JsonSerializable
{
    /**
     * @var bool
     */
    private $valid;
    /**
     * @var string
     */
    private $path;
    /**
     * @var array
     */
    private $errors = [];
    /**
     * @var array
     */
    private $fields = [];

    /**
     * FormData constructor.
     *
     * @param bool $valid
     * @param string $path
     */
    public function __construct(bool $valid, string $path)
    {
        $this->valid = $valid;
        $this->path = $path;
    }

    /**
     * @return bool
     */
    public function isValid(): bool
    {
        return $this->valid;
    }

    /**
     * @return string
     */
    public function getPath(): string
    {
        return $this->path;
    }

    /**
     * @param FormDataError $formDataError
     */
    public function addError(FormDataError $formDataError)
    {
        $this->errors[] = $formDataError;
    }

    /**
     * @return array
     */
    public function getErrors(): array
    {
        return $this->errors;
    }

    /**
     * @param mixed $field
     * @param FormData $formData
     */
    public function addField($field, FormData $formData)
    {
        $this->fields[$field] = $formData;
    }

    /**
     * @return array
     */
    public function getFields(): array
    {
        return $this->fields;
    }

    /**
     * Specify data which should be serialized to JSON.
     *
     * @see http://php.net/manual/en/jsonserializable.jsonserialize.php
     *
     * @return mixed data which can be serialized by <b>json_encode</b>,
     * which is a value of any type other than a resource
     *
     * @since 5.4.0
     */
    public function jsonSerialize()
    {
        return [
            'valid' => $this->isValid(),
            'path' => $this->getPath(),
            'errors' => $this->getErrors(),
            'fields' => $this->getFields(),
        ];
    }
}
