<?php

declare(strict_types=1);

namespace IssetBV\ErrorBundle\Service\Loader\Provider;

use InvalidArgumentException;
use IssetBV\ErrorBundle\Service\Error;
use IssetBV\ErrorBundle\Service\ErrorImpl;
use Symfony\Component\Finder\Finder;
use Symfony\Component\Yaml\Parser;

/**
 * Class YamlErrorProvider.
 */
class YamlErrorProvider implements ErrorProvider
{
    /**
     * @var string[]
     */
    private $checkPaths = [];
    /**
     * @var Parser
     */
    private $parser;

    /**
     * YamlErrorProvider constructor.
     *
     * @param string $base
     */
    public function __construct(string $base)
    {
        $this->parser = new Parser();
        $this->addCheckPath($base . '/src/**/Resources/config');
        $this->addCheckPath($base . '/config/errors');
        $this->addCheckPath($base . '/errors');
    }

    /**
     * @param string $path
     */
    public function addCheckPath(string $path)
    {
        $this->checkPaths[] = $path;
    }

    /**
     * @return Error[]
     */
    public function getErrors(): array
    {
        $errors = [];
        foreach ($this->checkPaths as $path) {
            try {
                $errors = array_merge($errors, $this->getForPath($path));
            } catch (InvalidArgumentException $e) {
            }
        }

        return $errors;
    }

    /**
     * @param string $path
     *
     * @throws InvalidArgumentException
     *
     * @return array
     */
    public function getForPath(string $path): array
    {
        $finder = (new Finder())->name('*errors.yml')->files();
        $return = [];
        $files = $finder->in($path);
        foreach ($files as $file) {
            $errors = $this->parser->parse(file_get_contents($file->getPathName()));
            $return = array_merge($return, $this->convertErrors($errors['errors']));
        }

        return $return;
    }

    /**
     * @param array $errors
     *
     * @return array
     */
    public function convertErrors(array $errors): array
    {
        $return = [];
        foreach ($errors as $errorKey => $error) {
            $return[$errorKey] = $this->convertError($errorKey, $error);
        }

        return $return;
    }

    /**
     * @param string $errorKey
     * @param array $error
     *
     * @return Error
     */
    public function convertError(string $errorKey, array $error): Error
    {
        return new ErrorImpl(
            $error['code'],
            array_key_exists('message', $error) ? $error['message'] : $errorKey,
            $error['description'],
            isset($error['http_code']) ? $error['http_code'] : 400
        );
    }
}
