<?php

declare(strict_types=1);

namespace IssetBV\ErrorBundle\Service\Loader;

use IssetBV\ErrorBundle\Service\Error;
use IssetBV\ErrorBundle\Service\Loader\Exception\ErrorMissingException;
use IssetBV\ErrorBundle\Service\Loader\Provider\ErrorProvider;

/**
 * Class FileCache.
 */
class FileCache implements ErrorLoader
{
    /**
     * @var Error[]|null
     */
    private $errors;
    /**
     * @var string
     */
    private $filePath;
    /**
     * @var ErrorProvider
     */
    private $errorProvider;

    /**
     * FileCache constructor.
     *
     * @param ErrorProvider $errorProvider
     * @param string $filePath
     */
    public function __construct(ErrorProvider $errorProvider, string $filePath)
    {
        $this->filePath = $filePath;
        $this->errorProvider = $errorProvider;
    }

    public function clearCache()
    {
        if (file_exists($this->filePath)) {
            unlink($this->filePath);
        }

        $this->errors = null;
    }

    /**
     * @return Error[]
     */
    public function loadAll(): array
    {
        if (null !== $this->errors) {
            return $this->errors;
        }

        if (file_exists($this->filePath)) {
            $this->errors = require $this->filePath;

            return $this->errors;
        }

        $this->errors = $this->errorProvider->getErrors();
        $data = '<?php return ' . var_export($this->errors, true) . ';';
        file_put_contents($this->filePath, $data);

        return $this->errors;
    }

    /**
     * @param string $error
     *
     * @throws ErrorMissingException
     *
     * @return Error
     */
    public function load(string $error): Error
    {
        $errors = $this->loadAll();
        if (!array_key_exists($error, $errors)) {
            throw new ErrorMissingException($error);
        }

        return $this->errors[$error];
    }
}
