<?php

declare(strict_types=1);

namespace IssetBV\CallbackBundle\Service\Callback\Processor;

use DateTime;
use Doctrine\Common\Persistence\ObjectManager;
use IssetBV\CallbackBundle\Entity\CallbackTry;
use IssetBV\CallbackBundle\Service\Callback\Response\ResponseData;
use IssetBV\CallbackBundle\Service\Callback\Response\ResponseHandlerInterface;
use IssetBV\Http\Client\ClientInterface;
use IssetBV\Http\Client\Exception\ConnectExceptionInterface;
use IssetBV\Http\Client\Options;
use IssetBV\Http\Client\OptionsInterface;

class HttpClientCallbackTryHandler implements CallbackTryHandlerInterface
{
    /**
     * @var ClientInterface
     */
    private $client;

    /**
     * @var ObjectManager
     */
    private $objectManager;
    /**
     * @var ResponseHandlerInterface
     */
    private $responseHandler;

    public function __construct(ObjectManager $objectManager, ClientInterface $client, ResponseHandlerInterface $responseHandler)
    {
        $this->client = $client;
        $this->objectManager = $objectManager;
        $this->responseHandler = $responseHandler;
    }

    public function handle(CallbackTry $callbackTry)
    {
        $callback = $callbackTry->getCallback();

        $request = $this->client->create(
            $callback->getMethod(),
            $callback->getUrl(),
            $callback->getHeaders(),
            $callback->getBody(),
            $callback->getVersion()
        );

        $triesCount = $callbackTry->getCallback()->getTriesCount();

        $options = new Options();
        $options->setOption(OptionsInterface::CONNECT_TIMEOUT, $triesCount);
        $options->setOption(OptionsInterface::READ_TIMEOUT, $triesCount + 1);

        try {
            $response = $this->client->call($request, $options);

            $callbackTry->setStatusCode($response->getStatusCode());
            $callbackTry->setResponse((string) $response->getBody());

            if ($this->responseHandler->validateResponse(ResponseData::fromArray($callbackTry->getCallback()->getResponse()), $response)) {
                $this->handleSuccess($callbackTry);
            } else {
                $this->handleFailed($callbackTry);
            }
        } catch (ConnectExceptionInterface $connectException) {
            $callbackTry->setResponse((string) $connectException->getMessage());
            $this->handleFailed($callbackTry);
        }

        $this->objectManager->flush();
    }

    public function handleSuccess(CallbackTry $callbackTry)
    {
        $callbackTry->setStatus(ProcessorInterface::STATUS_SUCCESS);
        $callbackTry->getCallback()->setSentDate(new DateTime());
    }

    private function handleFailed(CallbackTry $callbackTry)
    {
        $callbackTry->setStatus(ProcessorInterface::STATUS_FAILED);

        if ($callbackTry->getCallback()->reTry()) {
            $this->handleNewReTry($callbackTry);
        }
    }

    private function handleNewReTry(CallbackTry $callbackTry)
    {
        $countTries = $callbackTry->getCallback()->getTriesCount();

        switch ($countTries) {
            case 1:
                $dateSend = new DateTime('+5 minutes');
                break;
            case 2:
                $dateSend = new DateTime('+30 minutes');
                break;
            case 3:
                $dateSend = new DateTime('+60 minutes');
                break;
            default:
                $dateSend = new DateTime('+1 day');
                break;
        }

        $callbackTry = new CallbackTry(
            ProcessorInterface::TYPE_RETRY,
            ProcessorInterface::STATUS_PENDING,
            $callbackTry->getCallback(),
            $dateSend
        );

        $this->objectManager->persist($callbackTry);
    }
}
