<?php

declare(strict_types=1);

namespace IssetBV\CallbackBundle\Service\Callback\Processor;

use DateTime;
use Doctrine\Common\Persistence\ObjectManager;
use IssetBV\CallbackBundle\Entity\CallbackTry;
use IssetBV\Http\Client\ClientInterface;
use IssetBV\Http\Client\Options;
use IssetBV\Http\Client\OptionsInterface;

class HttpClientCallbackTryHandler implements CallbackTryHandlerInterface
{
    /**
     * @var ClientInterface
     */
    private $client;
    /**
     * @var OptionsInterface
     */
    private $options;
    /**
     * @var ObjectManager
     */
    private $objectManager;

    public function __construct(ObjectManager $objectManager, ClientInterface $client)
    {
        $this->client = $client;
        $this->options = new Options();
        $this->options->setOption(OptionsInterface::CONNECT_TIMEOUT, 3);
        $this->options->setOption(OptionsInterface::READ_TIMEOUT, 3);
        $this->objectManager = $objectManager;
    }

    public function handle(CallbackTry $callbackTry)
    {
        $callback = $callbackTry->getCallback();

        $request = $this->client->create(
            $callback->getMethod(),
            $callback->getUrl(),
            $callback->getHeaders(),
            $callback->getBody(),
            $callback->getVersion()
        );

        $response = $this->client->call($request, $this->options);

        $callbackTry->setStatusCode($response->getStatusCode());
        $callbackTry->setResponse((string) $response->getBody());

        if ($callback->getResponse()->validateResponse($response)) {
            $this->handleSuccess($callbackTry);
        } else {
            $this->handleFailed($callbackTry);
        }

        $this->objectManager->flush();
    }

    public function handleSuccess(CallbackTry $callbackTry)
    {
        $callbackTry->setStatus(ProcessorInterface::STATUS_SUCCESS);
        $callbackTry->getCallback()->setSentDate(new DateTime());
    }

    private function handleFailed(CallbackTry $callbackTry)
    {
        $callbackTry->setStatus(ProcessorInterface::STATUS_FAILED);

        if ($callbackTry->getCallback()->reTry()) {
            $this->handleNewReTry($callbackTry);
        }
    }

    private function handleNewReTry(CallbackTry $callbackTry)
    {
        $countTries = $callbackTry->getCallback()->getTriesCount();

        switch ($countTries) {
            case 1:
                $dateSend = new DateTime('+5 minutes');
                break;
            case 2:
                $dateSend = new DateTime('+30 minutes');
                break;
            case 3:
                $dateSend = new DateTime('+60 minutes');
                break;
            default:
                $dateSend = new DateTime('+1 day');
                break;
        }

        $callbackTry = new CallbackTry(
            ProcessorInterface::TYPE_RETRY,
            ProcessorInterface::STATUS_PENDING,
            $callbackTry->getCallback(),
            $dateSend
        );

        $this->objectManager->persist($callbackTry);
    }
}
