<?php

namespace IssetBv\CallbackBundle;

use IssetBv\CallbackBundle\Caller\Call;
use IssetBv\CallbackBundle\Entity\CallbackTry;
use IssetBv\CallbackBundle\Caller\Response;
use \DateTime;

class CallbackProcessor
{

    /**
     *
     * @var \IssetBv\CallbackBundle\Caller\Call
     */
    private $caller;

    /**
     * @var \Doctrine\ORM\EntityManager
     */
    private $doctrine;

    public function __construct($doctrine, Call $caller)
    {
        $this->caller   = $caller;
        $this->doctrine = $doctrine;
    }

    public function process($callback)
    {
        $response = $this->caller->call($callback);

        $this->saveResponse($callback, $response);

        $tries = $callback->getTries();
        $callback->setTries(++$tries);
        $callback->setNextTry(null);

        if ($this->handleStatusCode($callback, $response)) {
            $this->setSuccess($callback);
            return true;
        } else {
            $this->setFailed($callback);
            return false;
        }
    }

    private function handleStatusCode(Entity\Callback $callback, Response $response)
    {
        if ($callback->getResponseStatusCode() == null) {
            return $this->handleStatusCodeDefault($response);
        } else {
            return $this->handleStatusCodeCustom($callback, $response);
        }
    }

    private function handleStatusCodeCustom(Entity\Callback $callback, Response $response)
    {
        if ($response->getStatusCode() != $callback->getResponseStatusCode()) {
            return false;
        }

        $statusBody = $callback->getResponseBody();
        if (!empty($statusBody) && trim($response->getResponse()) != trim($statusBody)) {
            return false;
        }
        return true;
    }

    private function handleStatusCodeDefault(Response $response)
    {
        switch ($response->getStatusCode()) {
            case "202":
            case "204":
                return true;
                break;
            case "200":
                if (trim($response->getResponse(), " \t\n\r\0\x0B\"") == '[success]') {
                    return true;
                }
                break;
        }

        return false;
    }

    private function saveResponse(Entity\Callback $callback, Response $response)
    {
        $callbackTry = new CallbackTry();
        $callbackTry->setCallback($callback);
        $callbackTry->setReponseCode($response->getStatusCode());
        $callbackTry->setResponseBody($response->getResponse());
        $this->doctrine->persist($callbackTry);
    }

    private function setSuccess(Entity\Callback $callback)
    {
        $callback->setSuccess(true);
    }

    private function setFailed(Entity\Callback $callback)
    {
        $now = new DateTime();
        if ($callback->getTries() <= $callback->getTriesMax()) {
            $now->modify('+ ' . 5 * $callback->getTries() . ' minutes');
            $callback->setNextTry($now);
        }
    }

}