<?php

namespace IssetBv\CallbackBundle;

use IssetBv\CallbackBundle\Caller\Call;

class Callback
{

    /**
     * @var \Doctrine\ORM\EntityManager
     */
    private $doctrine;

    /**
     *
     * @var \IssetBv\CallbackBundle\Caller\Call
     */
    private $caller;

    public function __construct($doctrine, Call $caller)
    {
        $this->doctrine = $doctrine;
        $this->caller   = $caller;
    }

    public function add($url, $params = array(), $groupIdentifier = null, \DateTime $startDate = null, $responseStatusCode = null, $responseBody = null)
    {

        if (empty($startDate)) {
            $startDate = new \DateTime();
        }

        $callback = new Entity\Callback();
        $callback->setUrl($url);
        $callback->setGroupIdentifier($groupIdentifier);
        $callback->setNextTry($startDate);
        $callback->setResponseStatusCode($responseStatusCode);
        $callback->setResponseBody($responseBody);
        $this->doctrine->persist($callback);

        foreach ($params as $key => $value) {
            $callBackParam = new Entity\CallbackParam();
            $callBackParam->setKey($key);
            if (is_object($value) || is_array($value)) {
                $value = json_encode($value);
            }
            $callBackParam->setParam($value);
            $callBackParam->setCallback($callback);
            $this->doctrine->persist($callBackParam);
        }

        $this->doctrine->flush();
        return $callback->getId();
    }

    public function processFailed()
    {
        $min = new \DateTime();
        $min->modify("-3 days");

        //select "min"(id) from issetbv_callbacks where success = false and next_try IS NOT NULL group by group_identifier limit 10
        $idQb = $this->doctrine->createQueryBuilder();
        $ids  = $idQb->select("MIN(c.id) as id")
                ->from('IssetBvCallbackBundle:Callback', 'c')
                ->where('c.success = false')
                ->andWhere('c.nextTry IS NOT NULL')
                ->andWhere('c.nextTry <= :nexttry')
                ->andWhere('c.nextTry >= :min')
                ->andWhere('c.tries >= 1')
                ->setParameter(':nexttry', new \DateTime())
                ->setParameter(':min', $min)
                ->groupBy('c.groupIdentifier')
                ->getQuery()
                ->setMaxResults(50)
                ->getArrayResult()
        ;

        if (empty($ids)) {
            sleep(1);
            return;
        }

        $idData = array_map(function($array) {
                    return $array['id'];
                }, $ids);

        $qb      = $this->doctrine->createQueryBuilder();
        $results = $qb->select('c, p')->from('IssetBvCallbackBundle:Callback', 'c')
                ->leftJoin('c.params', 'p')
                ->where('c IN(:callbacks)')
                ->setParameter(':callbacks', $idData)
                ->getQuery()
                ->execute();
        ;

        foreach ($results as $callback) {
            if ($this->processCallback($callback)) {
                /* @var $callback \IssetBv\CallbackBundle\Entity\Callback */
                $groupIdentifier = $callback->getGroupIdentifier();
                if (empty($groupIdentifier)) {
                    continue;
                }
                $this->doctrine->flush();
                $this->handleGroupFrom($callback);
            }
        }
    }

    public function processNew()
    {
        $min = new \DateTime();
        $min->modify("-3 days");

        //select "min"(id) from issetbv_callbacks where success = false and next_try IS NOT NULL group by group_identifier limit 10
        $idQb = $this->doctrine->createQueryBuilder();
        $ids  = $idQb->select("MIN(c.id) as id")
                ->from('IssetBvCallbackBundle:Callback', 'c')
                ->where('c.success = false')
                ->andWhere('c.nextTry IS NOT NULL')
                ->andWhere('c.nextTry <= :nexttry')
                ->andWhere('c.nextTry >= :min')
                ->andWhere('c.tries = 0')
                ->setParameter(':nexttry', new \DateTime())
                ->setParameter(':min', $min)
                ->groupBy('c.groupIdentifier')
                ->getQuery()
                ->setMaxResults(50)
                ->getArrayResult()
        ;

        if (empty($ids)) {
            sleep(1);
            return;
        }

        $idData = array_map(function($array) {
                    return $array['id'];
                }, $ids);

        $qb      = $this->doctrine->createQueryBuilder();
        $results = $qb->select('c, p')->from('IssetBvCallbackBundle:Callback', 'c')
                ->leftJoin('c.params', 'p')
                ->where('c IN(:callbacks)')
                ->setParameter(':callbacks', $idData)
                ->getQuery()
                ->execute();
        ;

        foreach ($results as $callback) {
            if ($this->processCallback($callback)) {
                /* @var $callback \IssetBv\CallbackBundle\Entity\Callback */
                $groupIdentifier = $callback->getGroupIdentifier();
                if (empty($groupIdentifier)) {
                    continue;
                }
                $this->doctrine->flush();
                $this->handleGroupFrom($callback);
            }
        }
    }

    public function process()
    {

        $min = new \DateTime();
        $min->modify("-3 days");

        //select "min"(id) from issetbv_callbacks where success = false and next_try IS NOT NULL group by group_identifier limit 10
        $idQb = $this->doctrine->createQueryBuilder();
        $ids  = $idQb->select("MIN(c.id) as id")
                ->from('IssetBvCallbackBundle:Callback', 'c')
                ->where('c.success = false')
                ->andWhere('c.nextTry IS NOT NULL')
                ->andWhere('c.nextTry <= :nexttry')
                ->andWhere('c.nextTry >= :min')
                ->setParameter(':nexttry', new \DateTime())
                ->setParameter(':min', $min)
                ->groupBy('c.groupIdentifier')
                ->getQuery()
                ->setMaxResults(50)
                ->getArrayResult()
        ;

        if (empty($ids)) {
            sleep(1);
            return;
        }

        $idData = array_map(function($array) {
                    return $array['id'];
                }, $ids);


        $qb      = $this->doctrine->createQueryBuilder();
        $results = $qb->select('c, p')->from('IssetBvCallbackBundle:Callback', 'c')
                ->leftJoin('c.params', 'p')
                ->where('c IN(:callbacks)')
                ->setParameter(':callbacks', $idData)
                ->getQuery()
                ->execute();
        ;

        foreach ($results as $callback) {
            if ($this->processCallback($callback)) {
                /* @var $callback \IssetBv\CallbackBundle\Entity\Callback */
                $groupIdentifier = $callback->getGroupIdentifier();
                if (empty($groupIdentifier)) {
                    continue;
                }
                $this->doctrine->flush();
                $this->handleGroupFrom($callback);
            }
        }
    }

    public function handleGroupFrom($callbackOld)
    {
        $qb        = $this->doctrine->createQueryBuilder();
        $callbacks = $qb->select("c")
                ->from('IssetBvCallbackBundle:Callback', 'c')
                ->where('c.success = false')
                ->andWhere('c.nextTry IS NOT NULL')
                ->andWhere('c.id > :id')
                ->andWhere("c.groupIdentifier = :identifier")
                ->andWhere('c.nextTry <= :nexttry')
                ->setParameter(':nexttry', new \DateTime())
                ->setParameter(":id", $callbackOld->getId())
                ->setParameter(":identifier", $callbackOld->getGroupIdentifier())
                ->getQuery()
                ->execute()
        ;


        foreach ($callbacks as $callback) {
            $check = $this->processCallback($callback);
            $this->doctrine->flush();
            if (!$check) {
                break;
            }
        }
    }

    public function processCallback(\IssetBv\CallbackBundle\Entity\Callback $callback)
    {
        $response = $this->caller->call($callback);

        //add the try to the DB
        $callbackTry = new Entity\CallbackTry();
        $callbackTry->setCallback($callback);
        $callbackTry->setReponseCode($response->getStatusCode());
        $callbackTry->setResponseBody($response->getResponse());
        $this->doctrine->persist($callbackTry);

        $tries = $callback->getTries();
        $callback->setTries(++$tries);
        $callback->setNextTry(null);

        $statusCode = $callback->getResponseStatusCode();
        if (!empty($statusCode)) { // check if there is a custom status code
            if ($response->getStatusCode() != $statusCode) {
                return $this->setFailed($callback);
            }

            $statusBody = $callback->getResponseBody();
            if (!empty($statusBody) && trim($response->getResponse()) != trim($statusBody)) {
                return $this->setFailed($callback);
            }
            return $this->setSuccess($callback);
        } else if (//default check if the response should go through
                $response->getStatusCode() == '202' ||
                $response->getStatusCode() == '204' ||
                ($response->getStatusCode() == '200' && trim($response->getResponse(), " \t\n\r\0\x0B\"") == '[success]')
        ) {
            return $this->setSuccess($callback);
        }

        return $this->setFailed($callback);
    }

    private function setSuccess($callback)
    {
        $callback->setSuccess(true);
        return true;
    }

    private function setFailed($callback)
    {
        $now = new \DateTime();
        if ($callback->getTries() <= $callback->getTriesMax()) {
            $now->modify('+ ' . 5 * $callback->getTries() . ' minutes');
            $callback->setNextTry($now);
        }
        $this->doctrine->flush();
        return false;
    }

    public function retryCallback($retryDays)
    {
        if ($retryDays == -1) {
            $retryDays = 1461;
        }
        $dateTime  = new \DateTime();
        $dateTime->modify('- ' . $retryDays . ' day');

        $qb      = $this->doctrine->createQueryBuilder()
                ->select('c')
                ->from('IssetBvCallbackBundle:Callback', 'c')
                ->where('c.success <= :success')
                ->andWhere('c.nextTry > :date')
                ->setParameter(':success', 'false')
                ->setParameter(':date', $dateTime);
        $results = $qb->getQuery()->execute();


        if ($results) {
            foreach ($results as $result) {
                $result->setTries(0);
                $result->setNextTry(new \DateTime());
            }
            $this->doctrine->flush();
        }
    }

}
