<?php

namespace IssetBv\CallbackBundle;

use IssetBv\CallbackBundle\Caller\Call;

class Callback
{

    /**
     * @var \Doctrine\ORM\EntityManager
     */
    private $doctrine;
    private $cache = array();

    /**
     *
     * @var \IssetBv\CallbackBundle\Caller\Call
     */
    private $caller;

    public function __construct($doctrine, Call $caller)
    {
        $this->doctrine = $doctrine;
        $this->caller   = $caller;
    }

    public function add($url, $params = array(), $groupIdentifier = null, \DateTime $startDate = null)
    {

        if (empty($startDate)) {
            $startDate = new \DateTime();
        }

        $callback = new Entity\Callback();
        $callback->setUrl($url);
        $callback->setGroupIdentifier($groupIdentifier);
        $callback->setNextTry($startDate);
        $this->doctrine->persist($callback);

        foreach ($params as $key => $value) {
            $callBackParam = new Entity\CallbackParam();
            $callBackParam->setKey($key);
            $callBackParam->setParam($value);
            $callBackParam->setCallback($callback);
            $this->doctrine->persist($callBackParam);
        }

        $this->doctrine->flush();
    }

    public function process()
    {
        $qb = $this->doctrine->createQueryBuilder();
        $qb->select('c, p')->from('IssetBvCallbackBundle:Callback', 'c')
                ->join('c.params', 'p')
                ->where('c.success = false')
                ->andWhere('c.nextTry <= :nexttry')
                ->orderBy('c.id', 'ASC')
                ->setParameter(':nexttry', new \DateTime())
        ;

        $results = $qb->getQuery()->execute();

        foreach ($results as $callback) {
            $this->processCallback($callback);
        }
        $this->doctrine->flush();
    }

    public function processCallback(\IssetBv\CallbackBundle\Entity\Callback $callback)
    {
        //check if the callback can be done with the group
        $groupIdentifier = $callback->getGroupIdentifier();
        if (!empty($groupIdentifier)) {
            if (!$this->checkGroup($callback->getId(), $groupIdentifier)) {
                return;
            }
        }
        $response = $this->caller->call($callback);

        //add the try to the DB
        $callbackTry = new Entity\CallbackTry();
        $callbackTry->setCallback($callback);
        $callbackTry->setReponseCode($response->getStatusCode());
        $callbackTry->setResponseBody($response->getResponse());
        $this->doctrine->persist($callbackTry);

        $tries = $callback->getTries();
        $callback->setTries(++$tries);
        $callback->setNextTry(null);

        //check if the response should go through
        if (($response->getStatusCode() == '202' OR ($response->getStatusCode() == '200' && trim($response->getResponse()) == '[success]'))) {
            if (!empty($groupIdentifier)) {
                $this->cache[$groupIdentifier] = true;
            }
            $callback->setSuccess(true);
        } else {
            if (!empty($groupIdentifier)) {
                $this->cache[$groupIdentifier] = false;
            }

            $now = new \DateTime();
            if ($callback->getTries() <= $callback->getTriesMax()) {
                $now->modify('+ ' . 5 * $callback->getTries() . ' minutes');
                $callback->setNextTry($now);
            }
        }
    }

    public function checkGroup($id, $groupIdentifier)
    {
        if (!isset($this->cache[$groupIdentifier])) {
            $qb = $this->doctrine->createQueryBuilder()
                    ->select('c.groupIdentifier')
                    ->from('IssetBvCallbackBundle:Callback', 'c')
                    ->where('c.success = false')
                    ->andWhere('c.groupIdentifier = :groupIdentifier')
                    ->andWhere('c.id < :id')
                    ->setParameter(':id', $id)
                    ->setParameter(':groupIdentifier', $groupIdentifier)
                    ->setMaxResults(1);

            $results                       = $qb->getQuery()->getOneOrNullResult();
            $this->cache[$groupIdentifier] = empty($results);
        }
        return $this->cache[$groupIdentifier];
    }

    public function retryCallback($retryDays)
    {
        if ($retryDays == -1) {
            $retryDays = 1461;
        }
        $dateTime  = new \DateTime();
        $dateTime->modify('- ' . $retryDays . ' day');

        $qb      = $this->doctrine->createQueryBuilder()
                ->select('c')
                ->from('IssetBvCallbackBundle:Callback', 'c')
                ->where('c.success <= :success')
                ->andWhere('c.nextTry > :date')
                ->setParameter(':success', 'false')
                ->setParameter(':date', $dateTime);
        $results = $qb->getQuery()->execute();


        if ($results) {
            foreach ($results as $result) {
                $result->setTries(0);
                $result->setNextTry(new \DateTime());
            }
            $this->doctrine->flush();
        }
    }

}
