<?php

declare(strict_types=1);

namespace IssetBV\CacheBundle\Service\Redis;

use IssetBV\CacheBundle\Service\CacheInterface;
use Redis;

class RedisCache implements CacheInterface
{
    /**
     * @var Redis|null
     */
    private $client;
    /**
     * @var string
     */
    private $host;
    /**
     * @var int
     */
    private $port;
    /**
     * @var bool
     */
    private $persistent;

    public function __construct(string $host, int $port, bool $persistent)
    {
        $this->host = $host;
        $this->port = $port;
        $this->persistent = $persistent;
    }

    public function getClient(): Redis
    {
        if (null === $this->client) {
            $this->client = new Redis();
            if ($this->persistent) {
                $this->client->pconnect($this->host, $this->port);
            } else {
                $this->client->connect($this->host, $this->port);
            }
        }

        return $this->client;
    }

    /**
     * Check if a key exists in the cache.
     *
     * @param string $key
     *
     * @return bool
     */
    public function exists(string $key): bool
    {
        return $this->getClient()->exists($key);
    }

    /**
     * Get a key out of the cache.
     *
     * @param string $key
     *
     * @return mixed
     */
    public function get(string $key)
    {
        $data = $this->getClient()->get($key);
        if (false === $data) {
            return null;
        }

        return $data;
    }

    /**
     * Get multiple keys out of the cache at once.
     *
     * @param string[] $keys array of string keys
     *
     * @return array key value pair of the given keys. key has a NULL if not in cache
     */
    public function getMulti(array $keys): array
    {
        $data = $this->getClient()->mget($keys);
        $return = [];

        foreach ($data as $key => $value) {
            if (false !== $value) {
                $return[$keys[$key]] = $value;
            } else {
                $return[$keys[$key]] = null;
            }
        }

        return $return;
    }

    public function set(string $key, string $value)
    {
        $this->getClient()->set($key, $value);
    }

    public function expireSeconds(string $key, int $seconds)
    {
        $this->getClient()->expire($key, $seconds);
    }

    public function expireTimestamp(string $key, int $timestamp)
    {
        $this->getClient()->expireAt($key, $timestamp);
    }

    public function setMulti(array $keys)
    {
        $this->getClient()->mset($keys);
    }

    public function delete(string $key)
    {
        $this->getClient()->del([$key]);
    }

    public function deleteMulti(array $keys)
    {
        $this->getClient()->del($keys);
    }

    public function flush()
    {
        $this->getClient()->flushAll();
    }
}
