<?php

declare(strict_types=1);

namespace IssetBV\CacheBundle\Service\Predis;

use IssetBV\CacheBundle\Service\CacheInterface;
use Predis\Client;

class PredisCache implements CacheInterface
{
    /**
     * @var Client|null
     */
    private $client;
    /**
     * @var string
     */
    private $host;
    /**
     * @var int
     */
    private $port;
    /**
     * @var bool
     */
    private $persistent;

    public function __construct(string $host, int $port, bool $persistent)
    {
        $this->host = $host;
        $this->port = $port;
        $this->persistent = $persistent;
    }

    public function getClient(): Client
    {
        if (null === $this->client) {
            $this->client = new Client([
                'host' => $this->host,
                'port' => $this->port,
                'persistent' => $this->persistent,
            ]);
        }

        return $this->client;
    }

    /**
     * Check if a key exists in the cache.
     *
     * @param string $key
     *
     * @return bool
     */
    public function exists(string $key): bool
    {
        return 1 === $this->getClient()->exists($key);
    }

    public function get(string $key)
    {
        return $this->getClient()->get($key);
    }

    public function getMulti(array $keys): array
    {
        $data = $this->getClient()->mget($keys);
        $return = [];

        foreach ($data as $key => $value) {
            $return[$keys[$key]] = $value;
        }

        return $return;
    }

    public function set(string $key, string $value)
    {
        $this->getClient()->set($key, $value);
    }

    public function expireSeconds(string $key, int $seconds)
    {
        $this->getClient()->expire($key, $seconds);
    }

    public function expireTimestamp(string $key, int $timestamp)
    {
        $this->getClient()->expireat($key, $timestamp);
    }

    public function setMulti(array $keys)
    {
        $this->getClient()->mset($keys);
    }

    public function delete(string $key)
    {
        $this->getClient()->del([$key]);
    }

    public function deleteMulti(array $keys)
    {
        $this->getClient()->del($keys);
    }

    public function flush()
    {
        $this->getClient()->flushall();
    }
}
