<?php

namespace IssetBV\Util;

use ArrayIterator;
use Exception;
use IteratorAggregate;
use LogicException;
use Traversable;

/**
 * Class Optional.
 */
final class Optional implements IteratorAggregate
{
    /**
     * @var mixed
     */
    private $value;

    /**
     * @var array[]
     */
    private $emptyValues;

    /**
     * Optional constructor.
     *
     * @param mixed $value
     * @param array $emptyValues
     */
    private function __construct($value, array $emptyValues)
    {
        $this->emptyValues = $emptyValues;
        $this->value = $value;
    }

    /**
     * Executes the closure if a value is present.
     *
     * @param callable $closure
     */
    public function ifPresent(callable $closure)
    {
        if ($this->isPresent()) {
            $closure($this->value);
        }
    }

    /**
     * @return bool
     */
    public function isPresent()
    {
        return false === in_array($this->value, $this->emptyValues, true);
    }

    /**
     * @param mixed $object
     *
     * @return Optional
     */
    public static function of($object)
    {
        if (is_object($object)) {
            return new self($object, [null]);
        }

        if (is_string($object)) {
            return new self($object, [null, '']);
        }

        return new self($object, [null]);
    }

    /**
     * @return Optional
     */
    public static function emptyOption()
    {
        return new self(null, [null]);
    }

    /**
     * @param callable $closure
     *
     * @return Optional
     */
    public function filter(callable $closure)
    {
        if ($this->isPresent()) {
            $result = $closure($this->value);

            if ($result) {
                return self::of($this->value);
            } else {
                return self::emptyOption();
            }
        }

        return self::emptyOption();
    }

    /**
     * @param callable $mapper
     *
     * @return Optional
     */
    public function map(callable $mapper)
    {
        if ($this->isPresent()) {
            return self::of($mapper($this->value));
        }

        return self::emptyOption();
    }

    /**
     * @param callable $closure
     *
     * @return Optional
     */
    public function flatMap(callable $closure)
    {
        if ($this->isPresent()) {
            return $closure($this->value);
        }

        return self::emptyOption();
    }

    /**
     * @param mixed $value
     *
     * @return mixed
     */
    public function orElse($value)
    {
        if ($this->isPresent()) {
            return $this->value;
        } else {
            return $value;
        }
    }

    /**
     * @param string|Exception $exception
     *
     * @throws Exception
     *
     * @return mixed
     */
    public function orElseThrow($exception)
    {
        if ($this->isPresent()) {
            return $this->value;
        } else {
            if ($exception instanceof Exception) {
                throw $exception;
            } elseif (is_string($exception) && class_exists($exception)) {
                throw new $exception();
            } else {
                throw new LogicException('Can\'t throw ' . gettype($exception));
            }
        }
    }

    /**
     * @param callable $closure
     *
     * @return mixed
     */
    public function orElseGet(callable $closure)
    {
        if ($this->isPresent()) {
            return $this->value;
        } else {
            return $closure();
        }
    }

    /**
     * Retrieve an external iterator.
     *
     * @link http://php.net/manual/en/iteratoraggregate.getiterator.php
     *
     * @return Traversable An instance of an object implementing <b>Iterator</b> or
     * <b>Traversable</b>
     *
     * @since 5.0.0
     */
    public function getIterator()
    {
        return new ArrayIterator($this->isPresent() ? [$this->value] : []);
    }
}
