<?php

declare(strict_types=1);

namespace IssetBV\Date;

use DateInterval;
use DateTime;
use PHPUnit\Framework\TestCase;

class WindowTest extends TestCase
{
    /**
     * @test
     */
    public function it_should_construct_instances_equally()
    {
        $windows = [
            new Window(new DateTime('January 31 1991'), new DateTime('January 31 1992')),
            Window::createFromEndDateAndInterval(new DateTime('January 31 1992'), new DateInterval('P1Y')),
            Window::createFromStartDateAndInterval(new DateTime('January 31 1991'), new DateInterval('P1Y')),
        ];

        foreach ($windows as $windowA) {
            foreach ($windows as $windowB) {
                TestCase::assertEquals($windowA->getStartDate()->get(), $windowB->getStartDate()->get());
                TestCase::assertEquals($windowA->getEndDate()->get(), $windowB->getEndDate()->get());
            }
        }
    }

    /**
     * @expectedException \Assert\AssertionFailedException
     * @test
     */
    public function it_should_throw_an_exception_when_start_is_after_end()
    {
        new Window(new DateTime('January 31 1991'), new DateTime('January 30 1991'));
    }

    /**
     * @expectedException \Assert\AssertionFailedException
     * @test
     */
    public function it_should_throw_when_start_is_end()
    {
        new Window(new DateTime('January 31 1991'), new DateTime('January 31 1991'));
    }

    /**
     * @test
     */
    public function it_should_return_true_if_the_window_is_valid()
    {
        $window = new Window(new DateTime('January 31 1991'), new DateTime('NOW + 10 years'));

        TestCase::assertTrue($window->isValid());
    }

    /**
     * @test
     */
    public function it_should_return_false_if_the_window_is_invalid()
    {
        $window = new Window(new DateTime('January 31 1991'), new DateTime('January 31 2000'));

        TestCase::assertFalse($window->isValid());
    }

    /**
     * @test
     */
    public function it_should_return_the_start_and_end_date_correctly()
    {
        $empty = new Window(null, null);

        TestCase::assertTrue($empty->getStartDate()->isEmpty());
        TestCase::assertTrue($empty->getEndDate()->isEmpty());

        $start = new DateTime('January 31 1991');
        $end = new DateTime('January 31 2000');

        $nonEmpty = new Window($start, $end);

        TestCase::assertTrue($nonEmpty->getStartDate()->isDefined());
        TestCase::assertTrue($nonEmpty->getEndDate()->isDefined());

        TestCase::assertEquals($start, $nonEmpty->getStartDate()->get());
        TestCase::assertEquals($end, $nonEmpty->getEndDate()->get());
    }

    /**
     * @test
     */
    public function it_should_construct_from_intervals_correctly()
    {
        $interval = new DateInterval('P1Y');

        $window = Window::createFromInterval($interval);
        
        sleep(1);

        TestCase::assertTrue($window->isValid());
        TestCase::assertEquals((string) $window->getDuration()->get()->format('%d'), $interval->format('%d'));
    }
}
