<?php

declare(strict_types=1);

namespace IssetBV\Date;

use Assert\Assertion;
use Assert\AssertionFailedException;
use DateInterval;
use DateTimeImmutable;
use DateTimeInterface;
use PhpOption\None;
use PhpOption\Option;

/**
 * Class Window.
 *
 * @author Tim Fennis <tim@isset.nl>
 */
class Window
{
    /**
     * @var DateTimeInterface|null
     */
    private $startDate;

    /**
     * @var DateTimeInterface|null
     */
    private $endDate;

    /**
     * @param DateTimeInterface|null $startDate
     * @param DateTimeInterface|null $endDate
     *
     * @throws AssertionFailedException
     */
    public function __construct(DateTimeInterface $startDate = null, DateTimeInterface $endDate = null)
    {
        if ($startDate !== null && $endDate !== null) {
            Assertion::true($startDate->getTimestamp() < $endDate->getTimestamp());
        }

        $this->startDate = $startDate;
        $this->endDate = $endDate;
    }

    /**
     * @return Option
     */
    public function getDuration(): Option
    {
        return $this->startDate === null || $this->endDate === null
            ? None::create()
            : Option::fromValue($this->endDate->diff($this->startDate, true));
    }

    /**
     * @return Option
     */
    public function getStartDate(): Option
    {
        return Option::fromValue($this->startDate);
    }

    /**
     * @return Option
     */
    public function getEndDate(): Option
    {
        return Option::fromValue($this->endDate);
    }

    /**
     * @return bool
     */
    public function isValid()
    {
        $start = true;
        $end = true;

        if ($this->startDate !== null) {
            $start = $this->startDate->getTimestamp() < (new DateTimeImmutable())->getTimestamp();
        }

        if ($this->endDate !== null) {
            $end = $this->endDate->getTimestamp() > (new DateTimeImmutable())->getTimestamp();
        }

        return $start && $end;
    }

    /**
     * @param DateInterval $interval
     *
     * @throws AssertionFailedException
     *
     * @return Window
     */
    public static function createFromInterval(DateInterval $interval)
    {
        return self::createFromStartDateAndInterval(new DateTimeImmutable(), $interval);
    }

    /**
     * @param DateTimeInterface $start
     * @param DateInterval $interval
     *
     * @throws AssertionFailedException
     *
     * @return Window
     */
    public static function createFromStartDateAndInterval(DateTimeInterface $start, DateInterval $interval)
    {
        $end = DateTimeHelper::toImmutable($start)->add($interval);

        return new self($start, $end);
    }

    /**
     * @param DateTimeInterface $end
     * @param DateInterval $interval
     *
     * @throws AssertionFailedException
     *
     * @return Window
     */
    public static function createFromEndDateAndInterval(DateTimeInterface $end, DateInterval $interval)
    {
        $start = DateTimeHelper::toImmutable($end)->sub($interval);

        return new self($start, $end);
    }
}
