<?php

declare(strict_types=1);

namespace IssetBV\Util;

use PHPUnit\Framework\TestCase;

/**
 * Class PhpArrayMapTest.
 *
 * @author Tim Fennis <tim@isset.nl>
 */
class PhpArrayMapTest extends TestCase
{
    /**
     * @return array
     */
    public function implementationsProvider()
    {
        return [
            [new PhpArrayMap()],
        ];
    }

    /**
     * @test
     * @dataProvider implementationsProvider
     *
     * @param Map $map
     */
    public function it_should_return_empty_map_after_clear(Map $map)
    {
        $map->put('hoi', 'dit is een text');

        $map->clear();

        TestCase::assertEmpty($map);
    }

    /**
     * @test
     * @dataProvider implementationsProvider
     *
     * @param Map $map
     */
    public function it_should_replace_values_with_the_same_keys(Map $map)
    {
        $map->put('key', 'value1');
        $map->put('key', 'value2');

        TestCase::assertEquals('value2', $map['key']);
        TestCase::assertCount(1, $map);
    }

    /**
     * @test
     * @dataProvider implementationsProvider
     *
     * @param Map $map
     */
    public function it_should_compute_the_mapping_for_current_value(Map $map)
    {
        $map->put('key', 'value');
        $map->compute('key',
            // concat to self closure
            function ($key, $value) {
                return $value . $value;
            }
        );

        TestCase::assertEquals('valuevalue', $map['key']);
    }

    /**
     * @test
     * @dataProvider implementationsProvider
     *
     * @param Map $map
     */
    public function it_should_replace_the_value_if_the_key_has_no_value(Map $map)
    {
        $map->merge('key', 'value',
            function ($oldValue, $newValue) {
                return $oldValue . $newValue;
            }
        );

        TestCase::assertEquals('value', $map['key']);
    }

    /**
     * @test
     * @dataProvider implementationsProvider
     *
     * @param Map $map
     */
    public function it_should_merge_the_value_if_key_has_a_value(Map $map)
    {
        $map->put('key', 'Foo');
        $map->merge('key', 'Bar',
            function ($oldValue, $newValue) {
                return $oldValue . $newValue;
            }
        );

        TestCase::assertEquals('FooBar', $map['key']);
    }
}
