<?php

declare(strict_types=1);

namespace IssetBV\Util;

use ArrayIterator;
use InvalidArgumentException;

/**
 * Class PhpArrayMap.
 *
 * This class provides an implementation of the Map interface
 *
 * @author Tim Fennis <tim@isset.nl>
 */
final class PhpArrayMap extends AbstractMap
{
    /**
     * @var array
     */
    private $map;

    /**
     * PhpArrayMap constructor.
     */
    public function __construct()
    {
        $this->map = [];
    }

    /**
     * Removes all elements from this map.
     */
    public function clear()
    {
        $this->map = [];
    }

    /**
     * @param mixed $key
     *
     * @throws InvalidArgumentException if the key is not valid in this type of map
     *
     * @return bool
     */
    public function containsKey($key): bool
    {
        return array_key_exists(self::convertKey($key), $this->map);
    }

    /**
     * @param callable $action function($key, $value) BiFunction
     */
    public function forEach(callable $action)
    {
        foreach ($this->map as $key => $value) {
            $action($key, $value);
        }
    }

    /**
     * @param mixed $key
     *
     * @throws InvalidArgumentException if the key is not valid in this type of map
     *
     * @return mixed Returns the value associated with this key, if the key is not found this map may return NULL
     */
    public function get($key)
    {
        if ($this->containsKey($key)) {
            return $this->map[self::convertKey($key)];
        }

        return null;
    }

    /**
     * @param mixed $key
     * @param mixed $value
     *
     * @throws InvalidArgumentException if the key is not valid in this type of map
     *
     * @return mixed the previous value associated with key, or null if there was no mapping for key. (A null return can also indicate that the map previously associated null with key, if the implementation supports null values.)
     */
    public function put($key, $value)
    {
        $oldValue = $this->get($key);

        $this->map[self::convertKey($key)] = $value;

        return $oldValue;
    }

    /**
     * @param mixed $key
     *
     * @throws InvalidArgumentException if the key is not valid in this type of map
     *
     * @return mixed the previous value associated with key, or null if there was no mapping for key
     */
    public function remove($key)
    {
        $value = $this->get($key);

        unset($this->map[self::convertKey($key)]);

        return $value;
    }

    /**
     * @param callable $function
     */
    public function replaceAll(callable $function)
    {
        foreach ($this->map as $key => &$value) {
            $value = $function($key, $value);
        }
    }

    /**
     * @return int
     */
    public function size(): int
    {
        return count($this->map);
    }

    /**
     * @todo maybe replace with our own collection implementation
     *
     * @return array
     */
    public function values(): array
    {
        return array_values($this->map);
    }

    /**
     * @return ArrayIterator
     */
    public function getIterator()
    {
        return new ArrayIterator($this->map);
    }

    /**
     * @param mixed $key
     *
     * @throws InvalidArgumentException If the key cannot be used in this map
     *
     * @return mixed
     */
    private static function convertKey($key)
    {
        if (is_object($key)) {
            throw new InvalidArgumentException('cannot use object as key in ' . __CLASS__);
        } elseif (is_resource($key)) {
            throw new InvalidArgumentException('cannot use resource as key');
        }

        return $key;
    }
}
