<?php

declare(strict_types=1);

namespace IssetBV\Util;

use PHPUnit\Framework\TestCase;

/**
 * Class OptionalTest.
 *
 * @author Tim Fennis <tim@isset.nl>
 */
class OptionalTest extends TestCase
{
    public function emptyValueProvider()
    {
        return [
            [null],
            [''],
        ];
    }

    public function nonEmptyValueProvider()
    {
        return [
            ['someString'],
            [new \stdClass()],
            [false],
            [true],
            [0],
            [500],
        ];
    }

    /**
     * @dataProvider nonEmptyValueProvider
     * @param mixed $value
     */
    public function it_should_execute_callback_if_value_is_nonempty($value)
    {
        $optional = Optional::of($value);

        $mock = $this->getMockBuilder(\stdClass::class)
            ->setMethods(['testCallback'])
            ->getMock();

        $mock->expects(TestCase::once())
            ->method('testCallback')
            ->with(TestCase::equalTo($value));

        $optional->ifPresent([$mock, 'testCallback']);
    }

    /**
     * @test
     * @dataProvider emptyValueProvider
     * @param mixed $value
     */
    public function it_should_return_false_if_value_is_empty($value)
    {
        $optional = Optional::of($value);
        TestCase::assertFalse($optional->isPresent());
    }

    /**
     * @test
     * @dataProvider nonEmptyValueProvider
     * @param mixed $value
     */
    public function it_should_return_true_if_value_is_nonempty($value)
    {
        $optional = Optional::of($value);
        TestCase::assertTrue($optional->isPresent());
    }

    /**
     * @test
     */
    public function it_should_return_empty_optional_if_callable_returns_false()
    {
        $mock = $this->getMockBuilder(\stdClass::class)
            ->setMethods(['filter'])
            ->getMock();

        $mock->expects(TestCase::once())
            ->method('filter')
            ->willReturn(false);

        $optional = Optional::of('test');
        $optional = $optional->filter([$mock, 'filter']);

        TestCase::assertFalse($optional->isPresent());
    }

    /**
     * @test
     */
    public function it_should_return_an_optional_when_map_is_used()
    {
        $optional = Optional::of('test');
        $newOptional = $optional->map(
            function ($value) {
                return $value . $value;
            }
        );

        TestCase::assertEquals('testtest', $newOptional->orElse('some other value'));
    }

    /**
     * @test
     */
    public function it_should_execute_a_closure_when_flatMap_is_used()
    {
        $optional = Optional::of('test');
        $newOptional = $optional->flatMap(
            function ($value) {
                return Optional::of($value . $value);
            }
        );

        TestCase::assertEquals('testtest', $newOptional->orElse('some other value'));
    }

    /**
     * @test
     */
    public function it_should_return_the_argument_if_optional_is_empty()
    {
        $optional = Optional::empty();

        TestCase::assertEquals('or_else', $optional->orElse('or_else'));
    }

    /**
     * @test
     */
    public function it_should_throw_an_exception_if_optional_is_empty()
    {
        $optional = Optional::empty();

        $this->expectException(\Exception::class);

        $optional->orElseThrow(\Exception::class);
    }

    /**
     * @test
     * @dataProvider emptyValueProvider
     * @param mixed $value
     */
    public function it_should_execute_a_closure_if_optional_is_empty($value)
    {
        $optional = Optional::of($value);

        $mock = $this->getMockBuilder(\stdClass::class)
            ->setMethods(['testCallback'])
            ->getMock();

        $mock->expects(TestCase::once())
            ->method('testCallback');

        $optional->orElseGet([$mock, 'testCallback']);
    }
}
