<?php

declare(strict_types=1);

namespace IssetBV\Util;

use ArrayAccess;
use Countable;
use InvalidArgumentException;
use IteratorAggregate;

/**
 * Interface Map.
 *
 * @author Tim Fennis <tim@isset.nl>
 */
interface Map extends ArrayAccess, Countable, IteratorAggregate
{
    /**
     * Removes all elements from this map.
     */
    public function clear();

    /**
     * Attempts to compute a mapping for the specified key and its current mapped value (or null if there is no current mapping).
     *
     * @param mixed $key
     * @param callable $remappingFunction function($key $value) BiFunction
     *
     * If functional interfaces were a thing then $remappingFunction would implement the BiFunction interface
     *
     * @throws InvalidArgumentException if the key is not valid in this type of map
     *
     * @return mixed The computed value
     */
    public function compute($key, callable $remappingFunction);

    /**
     * @param mixed $key
     * @param callable $mappingFunction function($key) SimFunction
     *
     * @throws InvalidArgumentException if the key is not valid in this type of map
     *
     * @return mixed The computed value
     */
    public function computeIfAbsent($key, callable $mappingFunction);

    /**
     * @param mixed $key
     * @param callable $remappingFunction
     *
     * @throws InvalidArgumentException if the key is not valid in this type of map
     *
     * @return mixed The computed value
     */
    public function computeIfPresent($key, callable $remappingFunction);

    /**
     * @param mixed $key
     *
     * @throws InvalidArgumentException if the key is not valid in this type of map
     *
     * @return bool
     */
    public function containsKey($key): bool;

    /**
     * Performs the given action for each entry in this map until all entries have been processed or the action throws an exception.
     *
     * @param callable $action
     */
    public function forEach(callable $action);

    /**
     * @param mixed $key
     *
     * @throws InvalidArgumentException if the key is not valid in this type of map
     *
     * @return mixed Returns the value associated with this key, if the key is not found this map may return NULL
     */
    public function get($key);

    /**
     * @param mixed $key
     * @param mixed $default
     *
     * @throws InvalidArgumentException if the key is not valid in this type of map
     *
     * @return mixed Returns the value associated with the key or the default value
     */
    public function getOrDefault($key, $default);

//    public function hashCode(): int;

    /**
     * Returns true if this mapping is empty.
     *
     * @return bool
     */
    public function isEmpty(): bool;

//    public function keySet(): Set;

    /**
     * @param mixed $key
     * @param mixed $value
     * @param callable $remappingFunction function($oldValue, $newValue) BiFunction
     *
     * @throws InvalidArgumentException if the key is not valid in this type of map
     *
     * @return mixed The new value associated with the key
     */
    public function merge($key, $value, callable $remappingFunction);

    /**
     * @param mixed $key
     * @param mixed $value
     *
     * @throws InvalidArgumentException if the key is not valid in this type of map
     *
     * @return mixed the previous value associated with key, or null if there was no mapping for key. (A null return can also indicate that the map previously associated null with key, if the implementation supports null values.)
     */
    public function put($key, $value);

    /**
     * @param Map $map
     */
    public function putAll(Map $map);

    /**
     * @param mixed $key
     * @param mixed $value
     *
     * @throws InvalidArgumentException if the key is not valid in this type of map
     *
     * @return mixed Returns the current value if they key is not absent, returns null if the key was previously not available
     */
    public function putIfAbsent($key, $value);

    /**
     * @param mixed $key
     *
     * @throws InvalidArgumentException if the key is not valid in this type of map
     *
     * @return mixed the previous value associated with key, or null if there was no mapping for key
     */
    public function remove($key);

    /**
     * Removes the entry for the specified key only if it is currently mapped to the specified value.
     *
     * @param mixed $key
     * @param mixed $value
     *
     * @throws InvalidArgumentException if the key is not valid in this type of map
     *
     * @return bool
     */
    public function removePair($key, $value): bool;

    /**
     * Replaces the entry for the specified key only if it is currently mapped to some value.
     *
     * @param mixed $key
     * @param mixed $value
     *
     * @throws InvalidArgumentException if the key is not valid in this type of map
     *
     * @return mixed
     */
    public function replace($key, $value);

    /**
     * Replaces the entry for the specified key only if currently mapped to the specified value.
     *
     * @param mixed $key
     * @param mixed $oldValue
     * @param mixed $newValue
     *
     * @throws InvalidArgumentException if the key is not valid in this type of map
     *
     * @return bool
     */
    public function replacePair($key, $oldValue, $newValue): bool;

    /**
     * Replaces each entry's value with the result of invoking the given function on that entry until all entries have been processed or the function throws an exception.
     *
     * @param callable $function function($key, $value) BiFunction
     */
    public function replaceAll(callable $function);

    /**
     * @return int
     */
    public function size(): int;

    /**
     * @todo maybe replace with our own collection implementation
     *
     * @return array
     */
    public function values(): array;
}
