<?php

declare(strict_types=1);

namespace IssetBV\Util;

/**
 * Class AbstractMap.
 *
 * @author Tim Fennis <tim@isset.nl>
 */
abstract class AbstractMap implements Map
{
    public function offsetExists($offset)
    {
        return $this->containsKey($offset);
    }

    public function offsetGet($offset)
    {
        return $this->get($offset);
    }

    public function offsetSet($offset, $value)
    {
        $this->put($offset, $value);
    }

    public function offsetUnset($offset)
    {
        $this->remove($offset);
    }

    public function count()
    {
        return $this->size();
    }

    public function isEmpty(): bool
    {
        return $this->size() === 0;
    }

    public function compute($key, callable $remappingFunction)
    {
        $oldValue = $this->get($key);
        $newValue = $remappingFunction($key, $oldValue);

        if ($oldValue !== null) {
            if ($newValue !== null) {
                $this->put($key, $newValue);
            } else {
                $this->remove($key);
            }
        } else {
            if ($newValue !== null) {
                $this->put($key, $newValue);
            } else {
                return null; // added for clarity but falling back on the line below would be fine too
            }
        }

        return $newValue;
    }

    public function computeIfAbsent($key, callable $mappingFunction)
    {
        if ($this->get($key) === null) {
            $newValue = $mappingFunction($key);
            if ($newValue !== null) {
                $this->put($key, $newValue);
            }
        }

        return $this->get($key);
    }

    public function computeIfPresent($key, callable $remappingFunction)
    {
        if ($this->get($key) !== null) {
            $oldValue = $this->get($key);
            $newValue = $remappingFunction($key, $oldValue);
            if ($newValue !== null) {
                $this->put($key, $newValue);
            } else {
                $this->remove($key);
            }
        }

        return $this->get($key);
    }

    public function merge($key, $value, callable $remappingFunction)
    {
        $oldValue = $this->get($key);
        $newValue = ($oldValue === null) ? $value : $remappingFunction($oldValue, $value);

        if ($newValue === null) {
            $this->remove($key);
        } else {
            $this->put($key, $newValue);
        }

        return $this->get($key);
    }

    public function putAll(Map $map)
    {
        $map->forEach(
            function ($key, $value) {
                $this->put($key, $value);
            }
        );
    }

    public function putIfAbsent($key, $value)
    {
        $value = $this->get($key);
        if ($value === null) {
            $this->put($key, $value);
        }

        return $value;
    }

    public function getOrDefault($key, $default)
    {
        if ($this->containsKey($key)) {
            return $this->get($key);
        }

        return $default;
    }

    public function removePair($key, $value): bool
    {
        if ($this->containsKey($key) && $this->get($key) === $value) {
            $this->remove($key);

            return true;
        }

        return false;
    }

    public function replace($key, $value)
    {
        if ($this->containsKey($key)) {
            return $this->put($key, $value);
        }

        return null;
    }

    public function replacePair($key, $oldValue, $newValue): bool
    {
        if ($this->containsKey($key) && $this->get($key) === $oldValue) {
            $this->put($key, $newValue);

            return true;
        }

        return false;
    }
}
