<?php

declare(strict_types=1);

namespace IssetBV\Util;

use IssetBV\Util\Exception\NullPointerException;
use LogicException;
use Throwable;

/**
 * Class Optional.
 */
final class Optional
{
    /**
     * @var Optional
     */
    private static $EMPTY;

    /**
     * @var mixed
     */
    private $value;

    /**
     * @var array[]
     */
    private $emptyValues;

    /**
     * Optional constructor.
     *
     * @param mixed $value
     * @param array $emptyValues
     */
    private function __construct($value, array $emptyValues)
    {
        $this->emptyValues = $emptyValues;
        $this->value = $value;
    }

    /**
     * Executes the closure if a value is present.
     *
     * @param callable $closure
     */
    public function ifPresent(callable $closure)
    {
        if ($this->isPresent()) {
            $closure($this->value);
        }
    }

    /**
     * @return bool
     */
    public function isPresent(): bool
    {
        return false === in_array($this->value, $this->emptyValues, true);
    }

    /**
     * @param mixed $object
     *
     * @return Optional
     */
    public static function of($object): Optional
    {
        if (is_object($object)) {
            return new self($object, [null]);
        }

        if (is_string($object)) {
            return new self($object, [null, '']);
        }

        return new self($object, [null]);
    }

    /**
     * @param mixed $value
     *
     * @return Optional
     */
    public static function ofNullable($value)
    {
        try {
            return $value === null ? self::empty() : self::java_of($value);
        } catch (NullPointerException $e) {
            // This will never actually happen but since java_of can throw this exception PHPStorm wants it checked
            return self::empty();
        }
    }

    /**
     * @return Optional
     */
    public static function empty(): Optional
    {
        return self::$EMPTY ?: self::$EMPTY = new self(null, [null]);
    }

    /**
     * @param callable $predicate
     *
     * @return Optional
     */
    public function filter(callable $predicate): Optional
    {
        if ($this->isPresent()) {
            return $predicate($this->value)
                ? self::of($this->value)
                : self::empty();
        }

        return self::empty();
    }

    /**
     * @param callable $mapper
     *
     * @return Optional
     */
    public function map(callable $mapper): Optional
    {
        if (false === $this->isPresent()) {
            return self::empty();
        }

        return self::ofNullable($mapper($this->value));
    }

    /**
     * @param callable $closure
     *
     * @throws NullPointerException
     *
     * @return Optional
     */
    public function flatMap(callable $closure): Optional
    {
        if (false === $this->isPresent()) {
            return self::empty();
        }

        $result = $closure($this->value);

        if ($result === null) {
            throw new NullPointerException();
        }

        return $result;
    }

    /**
     * @param mixed $value
     *
     * @return mixed
     */
    public function orElse($value)
    {
        return $this->isPresent() ? $this->value : $value;
    }

    /**
     * @param string|Throwable $exception
     *
     * @throws Throwable
     *
     * @return mixed
     */
    public function orElseThrow($exception)
    {
        if ($this->isPresent()) {
            return $this->value;
        }
        if ($exception instanceof Throwable) {
            throw $exception;
        } elseif (is_string($exception) && class_exists($exception)) {
            throw new $exception();
        }
        throw new LogicException('Can\'t throw ' . gettype($exception));
    }

    /**
     * @param callable $closure
     *
     * @return mixed
     */
    public function orElseGet(callable $closure)
    {
        if ($this->isPresent()) {
            return $this->value;
        }

        return $closure();
    }

    /**
     * Indicates whether some other object is "equal to" this Optional. The
     * other object is considered equal if:
     *  - it is also an {@code Optional} and;
     *  - both instances have no value present or;
     *  - the present values are "equal to" each other via ===.
     *
     * @param mixed $object an object to be tested for equality
     *
     * @return bool
     */
    public function equals($object)
    {
        return $object === $this || ($object instanceof self && $object->value === $this->value);
    }

    /**
     * @param mixed $value
     *
     * @throws NullPointerException
     *
     * @return Optional
     */
    private static function java_of($value)
    {
        if (null === $value) {
            throw new NullPointerException('null is not a valid argument');
        }

        return new self($value, [null]);
    }
}
