<?php

declare(strict_types=1);

namespace IssetBV\Util;

use Countable;
use IssetBV\Util\Exception\UnsupportedOperationException;
use Iterator;

/**
 * Interface Collection.
 *
 * NOTE: This interface cannot implement @see \ArrayAccess because it does not support seeking at offsets!
 *
 * @author Tim Fennis <tim@isset.nl>
 */
interface Collection extends Iterable, Countable
{
    /**
     * Ensures that this collection contains the specified element.
     *
     * @param mixed $element
     *
     * @throws UnsupportedOperationException
     *
     * @return bool
     */
    public function add($element): bool;

    /**
     * Adds all of the elements in the specified collection to this collection.
     *
     * @param Collection $collection
     *
     * @throws UnsupportedOperationException
     *
     * @return bool
     */
    public function addAll(Collection $collection): bool;

    /**
     * Removes all of the elements from this collection.
     *
     * @throws UnsupportedOperationException
     *
     * @return void
     */
    public function clear();

    /**
     * Returns true if this collection contains the specified element.
     *
     * @param mixed $element
     *
     * @return bool
     */
    public function contains($element): bool;

    /**
     * Returns true if this collection contains all of the elements in the specified collection.
     *
     * @param Collection $collection
     *
     * @return bool
     */
    public function containsAll(Collection $collection): bool;

    /**
     * Compares the specified object with this collection for equality.
     *
     * @param mixed $element
     *
     * @return bool
     */
    public function equals($element): bool;

//    public function hashCode();

    /**
     * Returns true if this collection contains no elements.
     *
     * @return bool
     */
    public function isEmpty(): bool;

    /**
     * Returns an iterator over the elements in this collection.

     * @return Iterator
     */
    public function getIterator(): Iterator;

    /**
     * Removes a single instance of the specified element from this collection, if it is present (optional operation).

     * @param mixed $object
     *
     * @throws UnsupportedOperationException
     *
     * @return bool
     */
    public function removeElement($object): bool;

    /**
     * Removes all of this collection's elements that are also contained in the specified collection (optional operation).

     * @param Collection $collection
     *
     * @throws UnsupportedOperationException
     *
     * @return bool
     */
    public function removeAll(Collection $collection): bool;

    /**
     * Removes all of the elements of this collection that satisfy the given predicate.
     *
     * @param callable $filter
     *
     * @throws UnsupportedOperationException
     *
     * @return bool
     */
    public function removeIf(callable $filter): bool;

    /**
     * Retains only the elements in this collection that are contained in the specified collection (optional operation).

     * @param Collection $collection
     *
     * @throws UnsupportedOperationException
     *
     * @return bool
     */
    public function retainAll(Collection $collection): bool;

    /**
     * Returns the number of elements in this collection.
     *
     * @return int
     */
    public function size(): int;

    /**
     * Returns a sequential Stream with this collection as its source.
     *
     * @return Stream
     */
    public function stream(): Stream;

    /**
     * Returns an array containing all of the elements in this collection.
     *
     * @return array
     */
    public function toArray(): array;
}
