<?php

declare(strict_types=1);

namespace IssetBV\Util;

use Countable;
use InvalidArgumentException;
use IssetBV\Util\Exception\UnsupportedOperationException;
use Iterator;
use Traversable;

/**
 * Class ArrayList.
 *
 * @author Tim Fennis <tim@isset.nl>
 */
final class ArrayList implements OrderedCollection
{
    private $list;

    private $size;

    /**
     * ArrayList constructor.
     *
     * @param array|Traversable|null $source
     *
     * @throws InvalidArgumentException
     * @throws UnsupportedOperationException
     */
    public function __construct($source = null)
    {
        $this->size = 0;

        if (is_array($source)) {
            $this->list = \SplFixedArray::fromArray($source);
            $this->size = $this->list->getSize();
        } else {
            //this is slow

            $this->list = new \SplFixedArray(0);

            foreach ($source as $index => $element) {
                $this->set($index, $element);
            }
        }
    }

    public function ensureCapacity(int $minCapacity)
    {
        $current = $this->list->getSize();

        if ($minCapacity > $current) {
            $this->list->setSize($minCapacity);
        }
    }

    /**
     * Ensures that this collection contains the specified element.
     *
     * @param mixed $element
     *
     * @throws UnsupportedOperationException
     *
     * @return bool
     */
    public function add($element): bool
    {
        if ($this->size === $this->list->getSize()) {
            $this->ensureCapacity($this->size + 1);
        }

        $this->list[$this->size] = $element;

        return true;
    }

    /**
     * Adds all of the elements in the specified collection to this collection.
     *
     * @param Collection $collection
     *
     * @throws UnsupportedOperationException
     *
     * @return bool
     */
    public function addAll(Collection $collection): bool
    {
        return $this->insertAll($this->size, $collection);
    }

    /**
     * Removes all of the elements from this collection.
     *
     * @throws UnsupportedOperationException
     */
    public function clear()
    {
        if ($this->size > 0) {
            $this->size = 0;
        }
    }

    /**
     * Returns true if this collection contains the specified element.
     *
     * @param mixed $element
     *
     * @return bool
     */
    public function contains($element): bool
    {
        return $this->indexOf($element) !== -1;
    }

    /**
     * Returns true if this collection contains all of the elements in the specified collection.
     *
     * @param Collection $collection
     *
     * @return bool
     */
    public function containsAll(Collection $collection): bool
    {
        return $collection->stream()->allMatch(
            function ($element) {
                return $this->contains($element);
            }
        );
    }

    /**
     * Compares the specified object with this collection for equality.
     *
     * @param mixed $element
     *
     * @return bool
     */
    public function equals($element): bool
    {
        if (!$element instanceof self) {
            return false;
        }

        // @todo is this retarded?
        return $element->containsAll($this) && $this->contains($element);
    }

    /**
     * Returns true if this collection contains no elements.
     *
     * @return bool
     */
    public function isEmpty(): bool
    {
        return $this->size === 0;
    }

    /**
     * Returns an iterator over the elements in this collection.
     *
     * @return Iterator
     */
    public function getIterator(): Iterator
    {
        // @todo Is this 100x slower then it really needs to be?
        return new \ArrayIterator($this->list->toArray());
    }

    /**
     * Removes a single instance of the specified element from this collection, if it is present (optional operation).
     *
     * @param mixed $object
     *
     * @throws UnsupportedOperationException
     *
     * @return bool
     */
    public function removeElement($object): bool
    {
        $index = $this->indexOf($object);

        if ($index !== -1) {
            return $this->remove($index);
        }

        return false;
    }

    /**
     * Removes all of this collection's elements that are also contained in the specified collection (optional operation).
     *
     * @param Collection $collection
     *
     * @throws UnsupportedOperationException
     *
     * @return bool
     */
    public function removeAll(Collection $collection): bool
    {
        return $collection->stream()->anyMatch(
            function ($element) {
                return $this->removeElement($element);
            }
        );
    }

    /**
     * Removes all of the elements of this collection that satisfy the given predicate.
     *
     * @param callable $filter
     *
     * @throws UnsupportedOperationException
     *
     * @return bool
     */
    public function removeIf(callable $filter): bool
    {
        $isModified = false;

        foreach ($this->list as $index => $element) {
            if ($filter($element)) {
                $isModified |= $this->remove($index);
            }
        }

        return $isModified;
    }

    /**
     * Retains only the elements in this collection that are contained in the specified collection (optional operation).
     *
     * @param Collection $collection
     *
     * @throws UnsupportedOperationException
     *
     * @return bool
     */
    public function retainAll(Collection $collection): bool
    {
        // TODO: Implement retainAll() method.
    }

    /**
     * Returns the number of elements in this collection.
     *
     * @return int
     */
    public function size(): int
    {
        // TODO: Implement size() method.
    }

    /**
     * Returns a sequential Stream with this collection as its source.
     *
     * @return Stream
     */
    public function stream(): Stream
    {
        // TODO: Implement stream() method.
    }

    /**
     * Returns an array containing all of the elements in this collection.
     *
     * @return array
     */
    public function toArray(): array
    {
        // TODO: Implement toArray() method.
    }

    /**
     * Whether a offset exists.
     *
     * @link http://php.net/manual/en/arrayaccess.offsetexists.php
     *
     * @param mixed $offset <p>
     * An offset to check for.
     * </p>
     *
     * @return bool true on success or false on failure.
     * </p>
     * <p>
     * The return value will be casted to boolean if non-boolean was returned
     *
     * @since 5.0.0
     */
    public function offsetExists($offset)
    {
        // TODO: Implement offsetExists() method.
    }

    /**
     * Offset to retrieve.
     *
     * @link http://php.net/manual/en/arrayaccess.offsetget.php
     *
     * @param mixed $offset <p>
     * The offset to retrieve.
     * </p>
     *
     * @return mixed Can return all value types
     *
     * @since 5.0.0
     */
    public function offsetGet($offset)
    {
        // TODO: Implement offsetGet() method.
    }

    /**
     * Offset to set.
     *
     * @link http://php.net/manual/en/arrayaccess.offsetset.php
     *
     * @param mixed $offset <p>
     * The offset to assign the value to.
     * </p>
     * @param mixed $value <p>
     * The value to set.
     * </p>
     *
     * @since 5.0.0
     */
    public function offsetSet($offset, $value)
    {
        // TODO: Implement offsetSet() method.
    }

    /**
     * Offset to unset.
     *
     * @link http://php.net/manual/en/arrayaccess.offsetunset.php
     *
     * @param mixed $offset <p>
     * The offset to unset.
     * </p>
     *
     * @since 5.0.0
     */
    public function offsetUnset($offset)
    {
        // TODO: Implement offsetUnset() method.
    }

    /**
     * @param callable $consumer
     */
    public function forEach(callable $consumer)
    {
        // TODO: Implement forEach() method.
    }

    /**
     * Replaces the element at the specified position in this list with the specified element (optional operation).
     *
     * @param int $index
     * @param mixed $element
     *
     * @throws UnsupportedOperationException
     *
     * @return bool
     */
    public function set(int $index, $element): bool
    {
        // TODO: Implement set() method.
    }

    /**
     * Inserts all of the elements in the specified collection into this list at the specified position (optional operation).
     *
     * @param int $index
     * @param Collection $collection
     *
     * @throws UnsupportedOperationException
     *
     * @return bool
     */
    public function insertAll(int $index, Collection $collection): bool
    {
        // TODO: Implement insertAll() method.
    }

    /**
     * Returns the element at the specified position in this list.
     *
     * @param int $index
     *
     * @return mixed
     */
    public function get(int $index)
    {
        // TODO: Implement get() method.
    }

    /**
     * Returns the index of the first occurrence of the specified element in this list, or -1 if this list does not contain the element.
     *
     * @param $element
     *
     * @return int
     */
    public function indexOf($element): int
    {
        // TODO: Implement indexOf() method.
    }

    /**
     * Returns the index of the last occurrence of the specified element in this list, or -1 if this list does not contain the element.
     *
     * @param $element
     *
     * @return int
     */
    public function lastIndexOf($element): int
    {
        // TODO: Implement lastIndexOf() method.
    }

    /**
     * Removes the element at the specified position in this list (optional operation).
     *
     * @param int $index
     *
     * @throws UnsupportedOperationException
     *
     * @return bool
     */
    public function remove(int $index): bool
    {
        // TODO: Implement remove() method.
    }

    /**
     * Replaces each element of this list with the result of applying the operator to that element.
     *
     * @param callable $operator
     *
     * @throws UnsupportedOperationException
     */
    public function replaceAll(callable $operator)
    {
        // TODO: Implement replaceAll() method.
    }

    /**
     * Sorts this list according to the order induced by the specified Comparator.
     *
     * @param callable $comparator
     */
    public function sort(callable $comparator)
    {
        // TODO: Implement sort() method.
    }

    /**
     * Returns a view of the portion of this list between the specified fromIndex, inclusive, and toIndex, exclusive.
     *
     * @param int $fromIndex
     * @param int $toIndex
     *
     * @return OrderedCollection
     */
    public function subList(int $fromIndex, int $toIndex): OrderedCollection
    {
        // TODO: Implement subList() method.
    }

    /**
     * Count elements of an object.
     *
     * @link http://php.net/manual/en/countable.count.php
     *
     * @return int The custom count as an integer.
     * </p>
     * <p>
     * The return value is cast to an integer
     *
     * @since 5.1.0
     */
    public function count()
    {
        // TODO: Implement count() method.
    }
}
