<?php

namespace IssetBV\Util;

use LogicException;

/**
 * Class Optional.
 */
final class Optional
{
    /**
     * @var mixed
     */
    private $value;

    /**
     * @var array[]
     */
    private $emptyValues;

    /**
     * Optional constructor.
     *
     * @param mixed $value
     * @param array $emptyValues
     */
    private function __construct($value, array $emptyValues)
    {
        $this->emptyValues = $emptyValues;
        $this->value = $value;
    }

    /**
     * Executes the closure if a value is present.
     *
     * @param callable $closure
     */
    public function ifPresent(callable $closure)
    {
        if ($this->isPresent()) {
            $closure($this->value);
        }
    }

    /**
     * @return bool
     */
    public function isPresent() : bool
    {
        return false === in_array($this->value, $this->emptyValues, true);
    }

    /**
     * @param mixed $object
     *
     * @return Optional
     */
    public static function of($object) : Optional
    {
        if (is_object($object)) {
            return new self($object, [null]);
        }

        if (is_string($object)) {
            return new self($object, [null, '']);
        }

        return new self($object, [null]);
    }

    /**
     * @return Optional
     */
    public static function empty() : Optional
    {
        return new self(null, [null]);
    }

    /**
     * @param callable $closure
     *
     * @return Optional
     */
    public function filter(callable $closure) : Optional
    {
        if ($this->isPresent()) {
            $result = $closure($this->value);

            if ($result) {
                return self::of($this->value);
            } else {
                return self::empty();
            }
        }

        return self::empty();
    }

    /**
     * @param callable $mapper
     *
     * @return Optional
     */
    public function map(callable $mapper) : Optional
    {
        if ($this->isPresent()) {
            return self::of($mapper($this->value));
        }

        return self::empty();
    }

    /**
     * @param callable $closure
     *
     * @return Optional
     */
    public function flatMap(callable $closure) : Optional
    {
        if ($this->isPresent()) {
            return $closure($this->value);
        }

        return self::empty();
    }

    /**
     * @param mixed $value
     *
     * @return mixed
     */
    public function orElse($value)
    {
        if ($this->isPresent()) {
            return $this->value;
        } else {
            return $value;
        }
    }

    /**
     * @param string $class
     *
     * @throws LogicException
     *
     * @return mixed
     */
    public function orElseThrow($class)
    {
        if (false === class_exists($class)) {
            throw new LogicException('Class not found: ' . $class);
        }

        if ($this->isPresent()) {
            return $this->value;
        } else {
            throw new $class();
        }
    }

    /**
     * @param callable $closure
     *
     * @return mixed
     */
    public function orElseGet(callable $closure)
    {
        if ($this->isPresent()) {
            return $this->value;
        } else {
            return $closure();
        }
    }
}
