<?php

namespace Mediator\Client;

use Mediator\LoggerInterface;

class SoapPlacing implements PlacingInterface {

	protected $wsdl_path;
	protected $wsdl_cache;
	protected $wsdl_timeout;
	protected $id_prefix;
	protected $logger;

	public function __construct($config = array(), LoggerInterface $logger)
	{
		if (count($config) > 0)
		{
			$this->initialize($config);
		}

		$this->logger = $logger;

		ini_set('soap.wsdl_cache_enabled', $this->wsdl_cache);
		ini_set('default_socket_timeout', $this->wsdl_timeout);

		$this->connect();
	}

	/**
	 * Initiliaze config settings
	 *
	 * @param array $config
	 */
	public function initialize($config = array())
	{
		foreach ($config as $key => $val)
		{
			if (property_exists($this, $key))
			{
				$this->$key = $val;
			}
		}
	}

	/**
	 * Returns all available functions of the WSDL
	 *
	 * @return mixed
	 */
	public function getFunctions()
	{
		if ($this->client === FALSE)
		{
			return FALSE;
		}
		return $this->client->__getFunctions();
	}

	public function getById($placing_id)
	{
		$data = array(
			'PlacingId' => mb_convert_case($placing_id, MB_CASE_UPPER),
		);

		return $this->soapcall('GetPlacingById', $data);
	}

	public function register(\Mediator\Placing\PlacingData\RegisterPlacingDataInterface $data)
	{
		$placing_data = array(
			'AssetId' => $data->getId(),
			'PublishTo' => $data->getPublishProfile(),
			'PublishOn' => $data->getWindowStart(),
			'RemoveOn' => $data->getWindowEnd(),
			'AudioLanguages' => $data->getAudioLanguages(),
			'Name' => $data->getName(),
			'Description' => $data->getDescription(),
			'CreatePreview' => $data->hasPreview()
		);

		return $this->soapcall('RegisterPlacing', $placing_data);
	}

	public function update(\Mediator\Placing\PlacingData\UpdatePlacingData $data)
	{
		$placing_data = array('PlacingId' => $data->id,
			'PublishOn' => $data->getWindowStart(),
			'RemoveOn' => $data->getWindowEnd(),
		);

		if (isset($data->name))
		{
			$placing_data['Name'] = $data->name;
		}

		if (isset($data->description))
		{
			$placing_data['Description'] = $data->description;
		}

		return $this->soapcall('UpdatePlacing', $placing_data);
	}

	/**
	 * Tries to connect to Mediator
	 *
	 * @return bool
	 */
	protected function connect()
	{
		try
		{
			// need @ to catch error
			$this->client = @new \SoapClient($this->wsdl_path,
					array(
						'trace' => 1,
						'features' => SOAP_SINGLE_ELEMENT_ARRAYS,
						"exceptions" => 1)
			);
			return TRUE;
		}
		catch (\SoapFault $exception)
		{
			$this->_set_exception($exception);
			return FALSE;
		}
	}

	/**
	 * Do a call to Mediator
	 *
	 * @param string $method
	 * @param array $payload
	 * @return mixed
	 */
	protected function soapcall($method, $payload)
	{
		if ($this->client === FALSE)
			return FALSE;

		try
		{
			$result = $this->client->$method($payload);

			// make sure we get an 'OK!' from Mediator
			if ($result->Success !== TRUE AND isset($result->Errors))
			{
				$this->messages	 = $result->Errors;
				$result			 = FALSE;
			}
		}
		catch (\Exception $e)
		{
			$this->_set_exception($e);
			$result = FALSE;
		}

		// log every register action just in case
		if (strpos($method, 'Register') !== FALSE)
		{
			$this->logger->addNotice('Method called: ' . $method . ' Data: ' . json_encode($payload));
		}

		// always log errors
		if ($result === FALSE)
		{
			$this->logger->addError(json_encode($this->get_messages()));
		}

		$this->_set_debug();
		return $result;
	}

	/**
	 * Set errors
	 *
	 * @param object $exception
	 */
	protected function _set_exception($exception)
	{
		$this->debug['error']	 = $exception->faultstring;
		$this->debug['code']	 = $exception->faultcode;

		$this->logger->addError(json_encode(htmlentities($exception->faultstring) . ":" . $exception->faultcode));
	}

	/**
	 * Set debug info
	 */
	protected function _set_debug()
	{
		if ($this->client !== FALSE)
		{
			$this->debug['request']	 = $this->client->__getLastRequest();
			$this->debug['response'] = $this->client->__getLastResponse();
		}
	}

	protected function _set_message($messages)
	{
		$this->messages .= $messages;
	}

	public function get_messages()
	{
		return $this->messages;
	}

}
